import bpy
import math

bl_info = {
    "name": "KeyFrameMotionTools",
    "author": "BlenderCN极D创意",
    "version": (1, 1),
    "blender": (3, 6, 0),
    "location": "TimeLine > N Panel",
    "description": "选中包含动画的物体，在时间线N面板中，调节偏移的帧数进行偏移。也可对齐所有物体关键帧到时间线位置。",
    "category": "Animation"
}

# 添加一个偏移量属性，用于设定关键帧的偏移数量
bpy.types.Scene.setnum = bpy.props.IntProperty(name="数量", default=1)
bpy.types.Scene.keyframe_offset = bpy.props.IntProperty(name="偏移帧", default=0)
# 新增一个属性，用于保存复选框的状态
bpy.types.Scene.reverse_order = bpy.props.BoolProperty(name="反向", default=False)
bpy.types.Scene.all_channal = bpy.props.BoolProperty(name="包含所有通道", default=False)



#获取有序列表
def get_ordered_selected_objects():
    # 获取当前活动的物体
    active_object = bpy.context.active_object

    # 检查是否有活动物体
    if not active_object:
        return []
    # 获取当前选中的物体
    selected_objects = bpy.context.selected_objects
    
    if len(selected_objects)==0:
        return []
    # 根据与当前活动物体的距离对物体进行排序（按距离远近倒序排序）
    distance_object_list = [((obj.location - active_object.location).length, obj) for obj in selected_objects if obj != active_object]

    # 按照距离排序的顺序输出物体名称
    sorted_objects = [obj for distance, obj in sorted(distance_object_list, key=lambda x: -x[0])]

    # 将当前活动物体插入到列表最前面
    sorted_objects.insert(len(sorted_objects), active_object)
    
    # 存储具有动画数据的物体
    objects_with_animation = []
    
     # 检查每个物体是否具有动画数据，并将具有动画数据的物体添加到列表中
    for obj in sorted_objects:
        if obj.animation_data and obj.animation_data.action:
            objects_with_animation.append(obj)
            
    # 根据复选框的状态决定是否反转列表
    if bpy.context.scene.reverse_order:
        objects_with_animation.reverse()
    return objects_with_animation


#判断选中的轨道关键帧是否为一
def are_all_channels_single_keyframe(current_object):
    # 获取当前物体的所有 F-Curves
      # 获取当前物体的所有 F-Curves
    fcurves = current_object.animation_data.action.fcurves

    # 遍历所有 F-Curves，检查每个 F-Curve 是否至少有一个关键帧
    for fcurve in fcurves:
        if len([keyframe for keyframe in fcurve.keyframe_points if keyframe.select_control_point]) != 1:
            return False

    return True

#对齐所有通道到时间线
def align_fcurves_to_current_frame(obj,current_frame):
    # 遍历物体的每个 F-Curve
    for fcurve in obj.animation_data.action.fcurves:
        for keyframe_point in fcurve.keyframe_points:
            if keyframe_point.select_control_point:
            # 计算 F-Curve 在当前时间线上的偏移量
                offset = current_frame - int(fcurve.keyframe_points[0].co.x)
                # 将 F-Curve 上的所有关键帧点移动到当前时间线上的位置
                for keyframe_point in fcurve.keyframe_points:
                    keyframe_point.co.x += offset
                    if keyframe_point.interpolation in {'BEZIER', 'VECTOR'}:
                        keyframe_point.handle_left.x += offset
                        keyframe_point.handle_right.x += offset
#获取起始帧
def get_selected_keyframes_start_frame(current_object):
    # 获取当前物体的所有 F-Curves
    fcurves = current_object.animation_data.action.fcurves

    # 存储每个通道选择的关键帧的起始帧时间位置
    start_frames_time = []

    # 遍历所有 F-Curves，获取每个通道选择的关键帧的起始帧时间位置
    for fcurve in fcurves:
        selected_keyframes = [keyframe.co.x for keyframe in fcurve.keyframe_points if keyframe.select_control_point]
        if selected_keyframes:
            start_frames_time.append(selected_keyframes[0])
            
        else:
            return 0   
    # 返回最靠前的关键帧时间
    return min(start_frames_time)

#对齐
def move_keyframes(current_object,channal):
        # 获取当前物体的选中关键帧点
        selected_keyframes = [keyframe for fcurve in current_object.animation_data.action.fcurves for keyframe in fcurve.keyframe_points if keyframe.select_control_point]
        current_frame = bpy.context.scene.frame_current
        single_keyframe=are_all_channels_single_keyframe(current_object)
#        判断选择的是否为一个每通道关键帧
#        print(single_keyframe)a

        if single_keyframe:
            if channal:
                for keyframe in selected_keyframes:
                    keyframe.co.x = current_frame
                offset = current_frame - int(keyframe.co.x)
                # 更新贝塞尔手柄位置
                if keyframe.interpolation in {'BEZIER', 'VECTOR'}:
                    keyframe.handle_left.x += offset
                    keyframe.handle_right.x += offset
            else:
                # 移动选中的关键帧点
                for keyframe in selected_keyframes:
                    keyframe_offset=current_frame-keyframe.co.x
                    keyframe.co.x += keyframe_offset

                    # 更新贝塞尔手柄位置
                    if keyframe.interpolation in {'BEZIER', 'VECTOR'}:
                        keyframe.handle_left.x += keyframe_offset
                        keyframe.handle_right.x += keyframe_offset
        else:
            if channal:
                align_fcurves_to_current_frame(current_object,current_frame)
            else:
                
                keyframe_offset=int(current_frame-get_selected_keyframes_start_frame(current_object))
                
                # 更新贝塞尔手柄位置
                for keyframe in selected_keyframes:
                    # 移动选中的关键帧点
                    keyframe.co.x += keyframe_offset 
                    if keyframe.interpolation in {'BEZIER', 'VECTOR'}:
                        keyframe.handle_left.x += keyframe_offset
                        keyframe.handle_right.x += keyframe_offset
        # 更新关键帧所在帧数
        for fcurve in current_object.animation_data.action.fcurves:
            for keyframe in fcurve.keyframe_points:
                keyframe.co.x = round(keyframe.co.x)  # 确保关键帧所在帧数为整数

#偏移           
def offset_keyframe(current_object,keyframe_offset):
    # 获取当前物体的选中关键帧点
        selected_keyframes = [keyframe for fcurve in current_object.animation_data.action.fcurves for keyframe in fcurve.keyframe_points if keyframe.select_control_point]
        print(keyframe_offset)
        # 移动选中的关键帧点
        for keyframe in selected_keyframes:
            
            keyframe.co.x +=keyframe_offset

            # 更新贝塞尔手柄位置
            if keyframe.interpolation in {'BEZIER', 'VECTOR'}:
                keyframe.handle_left.x += keyframe_offset
                keyframe.handle_right.x += keyframe_offset
    

# 面板
class CustomKeyframePanel(bpy.types.Panel):
    bl_label = "关键帧工具箱"
    bl_idname = "OBJECT_PT_custom_keyframe_panel"
    bl_space_type = 'DOPESHEET_EDITOR'
    bl_region_type = 'UI'
    bl_category = "关键帧工具箱"

    def draw(self, context):
        layout = self.layout
        
        # 第一行
        row = layout.row(align=True)
        row.prop(context.scene, "setnum", text="间隔")
        row.prop(context.scene, "keyframe_offset", text="设置偏移帧")

        # 第二行
        row = layout.row(align=True)
        row.operator("object.move_keyframes_with_offset", text="偏移关键帧", icon="MODIFIER")
        row.prop(context.scene, "reverse_order", text="反向")

        # 第三行
        row = layout.row(align=True)
        row.operator("object.move_keyframes_to_time", text="对齐关键帧至时间线", icon="MODIFIER")
        row.prop(context.scene, "all_channal")
        

# 对齐
class OBJECT_OT_MoveKeyframesToTime(bpy.types.Operator):
    bl_idname = "object.move_keyframes_to_time"
    bl_label = "移动关键帧"
    bl_options = {'REGISTER', 'UNDO'}

    def move_keyframes_to_current_frame(self,context):
        # 获取当前帧数
        current_frame = bpy.context.scene.frame_current

        # 获取当前选中的物体
        selected_objects = get_ordered_selected_objects()

        # 遍历选中的物体并移动关键帧
        for obj in selected_objects:
            # 移动关键帧
            move_keyframes(obj,context.scene.all_channal)
            

    def execute(self, context):
        self.move_keyframes_to_current_frame(context)
        return {'FINISHED'}


# 偏移
class OBJECT_OT_MoveKeyframesWithOffset(bpy.types.Operator):
    bl_idname = "object.move_keyframes_with_offset"
    bl_label = "移动关键帧"
    bl_options = {'REGISTER', 'UNDO'}

    def move_keyframes_with_offset(self, context):
        # 获取当前选中的物体
        selected_objects = get_ordered_selected_objects()
        
        if context.scene.setnum<1:
            return
        # 遍历选中的物体并移动关键帧
        for i, obj in enumerate(selected_objects):
            
            offset_num=math.floor(i/context.scene.setnum)
            offset_keyframe(obj, context.scene.keyframe_offset*offset_num)

    def execute(self, context):
        self.move_keyframes_with_offset(context)
        return {'FINISHED'}

classes = (
    CustomKeyframePanel,
    OBJECT_OT_MoveKeyframesWithOffset,
    OBJECT_OT_MoveKeyframesToTime
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)

def unregister():
    for cls in classes:
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()
