

bl_info = {
    "name": "CameraExtra",
    "author": "BlenderCN极D创意",
    "version": (2, 4, 1),
    "blender": (3, 6, 0),
    "location": "N面板-视图",
    "description": "可以为场景中的每一个摄像机设定独立的分辨率，独立的保存路径，一键为相机添加相机目标，一键设置希区柯克变焦效果，同时支持项目转移，目前只能单帧渲染，可惜不能用于动画渲染。",
    "category": "Camera",
    "latest_version ": (2, 4, 1),
    "wiki_url": "https://www.blendercn.org/19665.html",
    "support": "COMMUNITY",
}

import bpy
import mathutils
import requests
import json
from bpy.types import UIList
from bpy.props import BoolProperty

def get_all_collections_with_children(collection):
    collections = [collection]
    for child_collection in collection.children:
        collections.extend(get_all_collections_with_children(child_collection))
    return collections

def switch_collections(active_camera):
    # 获取当前视图层
    view_layer = bpy.context.view_layer

    # 获取视图层中所有层集合及其子集
    all_collections = []
    for layer_collection in view_layer.layer_collection.children:
        all_collections.extend(get_all_collections_with_children(layer_collection))
    if active_camera.camera_collections:
        for collection in all_collections:
            collection.exclude = False
            for camera_collection in active_camera.camera_collections:
                if camera_collection.name == collection.name:
                    collection.exclude = True

def set_status_bar_message(message):
    # 将消息添加到信息窗口中
    bpy.context.window_manager.popup_menu(message_func, title=message, icon='INFO')


def message_func(self, context):
    ...

def find_camera_by_name(camera_name):
    obj = find_obj(camera_name)
    # 遍历场景中的所有对象
    if obj and obj.type == 'CAMERA':
        return obj
    # 如果未找到相机对象，则返回None
    return None


def find_obj(name):
    obj = bpy.data.objects.get(name)
    if obj:
        return obj
    else:
        return None

def get_object_collection_name(obj):
    # 获取物体所在的所有集合
    collections = obj.users_collection

    # 返回第一个集合的名称（您可以根据需要进一步处理多个集合的情况）
    if collections:
        return collections[0].name
    else:
        return None
def set_collection_exclude_in_view_layer(collection_name, exclude_value):
    # 获取当前视图层
    view_layer = bpy.context.view_layer
    view_layer.layer_collection.children[collection_name].exclude = exclude_value

prev_active_camera = None

# 动态更新分辨率
def update_resolution(scene):
    global prev_active_camera
    active_camera = scene.camera
    if active_camera != prev_active_camera:
        for i, active_item in enumerate(scene.camera_list):
            if active_item.name_camera == active_camera.name:
                scene.world=active_item.world
                if active_item.camera_collections:
                    switch_collections(active_item)
                c = get_object_collection_name(active_camera)
                if c and c != "Scene Collection":
                    set_collection_exclude_in_view_layer(c, False)
                scene.camera_list_active_index = i
                prev_active_camera=active_camera
        if active_camera.get("resolution_x") and active_camera.get("resolution_y"):
            # 更新场景的输出分辨率
            scene.render.resolution_x = active_camera["resolution_x"]
            scene.render.resolution_y = active_camera["resolution_y"]
        if active_camera.get('filepath'):
            scene.render.filepath = active_camera.get('filepath')



def create_empty_at_camera_position(context, camera, name, type):
    targetname = camera.name + name
    # 删除同名
    existing_empty = bpy.data.objects.get(targetname)
    if existing_empty:
        return existing_empty
    # 获取相机的朝向向量
    camera_direction = camera.matrix_world.to_quaternion() @ mathutils.Vector((0, 0, -1))
    # 将空物体移动到相机的朝向方向1.5米处
    empty_location = context.scene.cursor.location
    # 创建空物体并将其放置在相机位置
    bpy.ops.object.empty_add(location=empty_location)
    empty_object = context.active_object
    empty_object.name = targetname
    # 设置空物体的尺寸
    empty_object.empty_display_size = 0.1
    # 设置空物体的显示类型为立方体
    empty_object.empty_display_type = type
    camera.select_set(True)
    empty_object.select_set(False)
    # 返回创建的空物体
    return empty_object

def add_camera(context):
    # 获取当前活动相机的名称
    cameras = [obj for obj in context.selected_objects if obj.type == 'CAMERA']
    if context.scene.camera_list:
        item=context.scene.camera_list[context.scene.camera_list_active_index]
        if not find_camera_by_name(item.name_camera):
            active_object = bpy.context.active_object
            if active_object.type == 'CAMERA':
                item.name_camera=active_object.name
                item.name=f"{active_object.get('world')}_{active_object.name}_{active_object.get('resolution_x')}*{active_object.get('resolution_y')}:{active_object.get('filepath') if active_object.get('filepath') else ''}"
                return

    for active_camera in cameras:
        # 获取当前场景输出的分辨率
        resolution_x = context.scene.render.resolution_x
        resolution_y = context.scene.render.resolution_y

        # 设置相机对象的自定义属性
        active_camera["resolution_x"] = resolution_x
        active_camera["resolution_y"] = resolution_y

        item_name_c = [item.name_camera for item in context.scene.camera_list]
        print(active_camera.data)
        if active_camera.name in item_name_c:
            context.scene.camera_list_active_index = item_name_c.index(active_camera.name)
            active_item = context.scene.camera_list[context.scene.camera_list_active_index]
        else:
            active_item = context.scene.camera_list.add()
            active_item.name_camera = active_camera.name
        if context.scene.world:
            active_item.world = context.scene.world
            active_camera["world"]=active_item.world.name
        else:
            active_camera["world"]=''
            active_item.world = None
        if active_camera.get('filepath'):
            filepath=active_camera.get('filepath')
        else:
            filepath=""

        camera_name = f"{active_camera.get('world')}_{active_camera.name}_{active_camera.get('resolution_x')}*{active_camera.get('resolution_y')}:{filepath}"

        # 添加相机名称到列表中，并设置分辨率
        active_item.name = camera_name
        active_item.resolution_x = resolution_x
        active_item.resolution_y = resolution_y

        bpy.context.view_layer.objects.active = active_camera


def add_track_to_constraint(camera_name, target_name):
    # 获取相机对象
    camera_obj = bpy.data.objects[camera_name]
    camera_obj.select_set(True)

    # 创建"Track To"约束
    bpy.ops.object.constraint_add(type='TRACK_TO')

    # 设置约束的目标对象
    target_obj = bpy.data.objects[target_name]
    bpy.context.object.constraints["Track To"].target  = target_obj


def add_constraints_to_camera_and_empty(camera, empty):
    if camera and empty:
        # 删除相机上之前的约束
        for constraint in camera.constraints:
            camera.constraints.remove(constraint)

        # 删除空物体上之前的约束
        for constraint in empty.constraints:
            empty.constraints.remove(constraint)

        # 为相机添加跟随约束，目标是空物体
        camera_constraint = camera.constraints.new(type='TRACK_TO')
        camera_constraint.target = empty
        camera_constraint.track_axis = 'TRACK_NEGATIVE_Z'
        camera_constraint.up_axis = 'UP_Y'

        # 为空物体添加复制旋转约束，目标是相机
        # empty_constraint = empty.constraints.new(type='COPY_ROTATION')
        # empty_constraint.target = camera
        bpy.context.view_layer.objects.active = camera


def get_camera_direction(camera):
    # 获取相机的世界坐标朝向（四元数）
    camera_quaternion = camera.matrix_world.to_quaternion()
    # 将四元数转换为欧拉角（角度）
    camera_euler = camera_quaternion.to_euler()
    return camera_euler


def remove_camera_target(camera):
    # 记录相机当前的世界坐标朝向
    camera_direction = get_camera_direction(camera)
    # 删除相机的标准跟随约束
    track_to_constraint = None
    for constraint in camera.constraints:
        if constraint.type == 'TRACK_TO' and constraint.target and constraint.target.name.endswith("_target"):
            track_to_constraint = constraint
            break

    if track_to_constraint:
        camera.constraints.remove(track_to_constraint)

    # 恢复相机之前的朝向
    camera.rotation_euler = camera_direction

    # 删除相机的目标
    target_name = camera.name + "_target"
    target_object = bpy.data.objects.get(target_name)
    if target_object:
        bpy.data.objects.remove(target_object, do_unlink=True)


def has_driver(camera_obj):
    cam_data = camera_obj.data
    # 检查相机焦距是否有驱动器
    if cam_data.animation_data and cam_data.animation_data.drivers:
        for driver in cam_data.animation_data.drivers:
            # 检查驱动器的数据路径是否为 "lens"，即焦距属性
            if driver.data_path == "lens":
                return True
    else:
        return False


def has_Track(camera_obj):
    for constraint in camera_obj.constraints:
        if constraint.type == 'TRACK_TO' and constraint.target and constraint.target.name.endswith("_target"):
            return True
        else:
            return False
def fetch():
    url = "https://service-azrgylmp-1255427755.sh.apigw.tencentcs.com/release/"
    response = requests.get(url)
    res=response.text
    bl_info["latest_version" ]= tuple(json.loads(res).get("version"))
    bl_info["downloads"] = json.loads(res).get("download_url")
    return

# 注册
def reg_handlers():
    bpy.app.handlers.frame_change_pre.append(update_resolution)

# 卸载
def unreg_handlers():
    bpy.app.handlers.frame_change_pre.remove(update_resolution)

class CollectionItem(bpy.types.PropertyGroup):
    name: bpy.props.StringProperty()


class CameraExtraPropGroup(bpy.types.PropertyGroup):
    camera=bpy.props.PointerProperty(type=bpy.types.Camera)
    name: bpy.props.StringProperty()
    name_camera: bpy.props.StringProperty()
    filepath: bpy.props.StringProperty()
    resolution_x: bpy.props.IntProperty(name="X分辨率", min=4)
    resolution_y: bpy.props.IntProperty(name="Y分辨率", min=4)
    camera_collections:bpy.props.CollectionProperty(type=CollectionItem)
    world:bpy.props.PointerProperty(type=bpy.types.World)

    # 定义一个方法，接收上下文参数
    def setup_resolution(self, context):
        # 使用上下文来获取属性
        self.resolution_x = context.scene.render.resolution_x
        self.resolution_y = context.scene.render.resolution_y

class CameraExtraPanel(bpy.types.Panel):
    bl_label = f"相机扩展{bl_info['version']}"
    bl_idname = "SCENE_PT_camera_extra"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "View"
    use_pin=True
    bl_order = 0

    def draw(self, context):
        layout = self.layout

        row = layout.row()
        row.scale_y = 1.5
        row.operator("scene.align_view_to_camera", text="创建视口相机")
        row.operator("scene.add_camera", text="更新相机列表")

        # 获取当前选择的物体
        selected_objects = bpy.context.selected_objects
        if len(selected_objects) == 1 and selected_objects[0].type == 'CAMERA':
            # 获取选择的相机
            camera = selected_objects[0]
            # 检查相机是否有自定义属性
            row = layout.row()
            if has_Track(camera):
                # 如果相机有自定义属性，则显示按钮
                row.operator("scene.camera_target", text="移除相机目标", icon="CANCEL").func = ""
            else:
                # 如果相机没有自定义属性，则不显示按钮
                row.operator("scene.camera_target", text="添加相机目标").func = "add"

            # 检查相机是否有自定义属性
            if has_driver(camera):
                # 如果相机有自定义属性，则显示按钮
                row.operator("scene.add_lens_driver", text="移除变焦", icon="CANCEL").func = ""
            else:
                # 如果相机没有自定义属性，则不显示按钮
                row.operator("scene.add_lens_driver", text="希区柯克变焦").func = "add"

        row = layout.row()
        row.operator("scene.read_camera", text="读取相机分辨率")

        row = layout.row()
        if context.scene.update_switch:
            row.operator("scene.update_switch_op", text="已开启动态更新", icon='CHECKBOX_HLT', emboss=True)
        else:
            row.operator("scene.update_switch_op", text="已关闭动态更新", icon='CHECKBOX_DEHLT')

        if context.scene.camera_list:
            row = layout.row()
            row.operator("scene.camera_file", text="记录输出路径", icon='FILE_FOLDER')
            row.scale_y = 1.5

            layout.separator()
            row = layout.row()
            row.prop(context.scene, "reverse_order", text="删除相机")
            row.prop(context.scene,"check_render", text="渲染区间")
            row.prop(context.scene,"check_preview", text="预览区间")
            row = layout.row()
            row.operator("scene.camera_collection", text="记录视图层").func="add"
            row.operator("scene.camera_collection", text="记录为默认").func="default"
            row.operator("scene.camera_collection", text="恢复默认").func="set_default"

            list_widget = layout.box().column()
            for idx, camera in enumerate(context.scene.camera_list):
                row = list_widget.row()
                row.operator("scene.set_active_camera", text=camera.name,
                             emboss=(idx == context.scene.camera_list_active_index)).idx = idx
                op = row.operator("scene.add_camera_marker", text='', icon='MARKER_HLT')
                op.idx = idx
                op.name = camera.name_camera

                row.operator("scene.remove_camera", text="", icon="PANEL_CLOSE").idx = idx

        row=layout.row()
        if bl_info["latest_version"]!=bl_info["version"]:
            row.label(text="Camera_Extra当前有新版本可用{}".format(bl_info["latest_version"]))


class CameraCollection(bpy.types.Operator):
    bl_idname = "scene.camera_collection"
    bl_label = "CameraCollection"
    func: bpy.props.StringProperty()

    def add_unique_collection_name(self, extra_obj, name):
        for collection_item in extra_obj.camera_collections:
            if collection_item.name == name:
                return  # 已存在相同名称，不需记录
        new_collection_item = extra_obj.camera_collections.add()
        new_collection_item.name = name

    def execute(self, context):
        # 获取当前视图层
        all_collections = []
        selected_objects = bpy.context.selected_objects
        for layer_collection in bpy.context.view_layer.layer_collection.children:
            # 获取视图层中所有层集合及其子集
            all_collections.extend(get_all_collections_with_children(layer_collection))
        if self.func == "add":
            camera_collections=[]
            seted_camera_list=[]
            cameras = [obj for obj in selected_objects if obj.type == 'CAMERA']
            for camera_extra in context.scene.camera_list:
                for active_camera in cameras:
                    if camera_extra.name_camera == active_camera.name:
                        active_camera["camera_collections"]=[]
                        seted_camera_list.append(active_camera.name)
                        camera_extra.camera_collections.clear()
                        for collection in all_collections:
                            if collection.exclude:
                                self.add_unique_collection_name(camera_extra,collection.name)
                                camera_collections.append(collection.name)
            active_camera["camera_collections"] = camera_collections
            self.report({'INFO'}, '列表中的{}记录成功'.format(seted_camera_list))

        elif self.func == "default":
            default_c=[]
            for collection in all_collections:
                if collection.exclude:
                    default_c.append(collection.name)
            context.scene['default_c'] = default_c
            self.report({'INFO'}, '设置默认集合成功')
        elif self.func == "set_default":
            if not context.scene.get('default_c'):
                self.report({'INFO'}, '暂无默认集合记录')
                return {'FINISHED'}
            for collection in all_collections:
                if collection.name in context.scene.get('default_c'):
                    collection.exclude = True
                    continue
                collection.exclude = False
            camera_collections = []
            seted_camera_list = []
            cameras = [obj for obj in selected_objects if obj.type == 'CAMERA']
            for camera_extra in context.scene.camera_list:
                for active_camera in cameras:
                    if camera_extra.name_camera == active_camera.name:
                        active_camera["camera_collections"] = []
                        seted_camera_list.append(active_camera.name)
                        camera_extra.camera_collections.clear()
                        for collection in all_collections:
                            if collection.exclude:
                                self.add_unique_collection_name(camera_extra, collection.name)
                                camera_collections.append(collection.name)
            active_camera["camera_collections"] = camera_collections
            self.report({'INFO'}, '列表中的{}恢复为默认集合状态'.format(seted_camera_list))
        return {'FINISHED'}

class CameraTargetOperator(bpy.types.Operator):
    bl_idname = "scene.camera_target"
    bl_label = "Camera Target"
    func: bpy.props.StringProperty()

    def execute(self, context):
        # 获取当前选中的对象
        active_object = bpy.context.active_object
        if active_object and active_object.type == 'CAMERA':
            # 创建空物体并添加约束
            if self.func == "add":
                empty_object = create_empty_at_camera_position(context, active_object, "_target", "CUBE")
                empty_object.show_in_front = True
                add_constraints_to_camera_and_empty(active_object, empty_object)

            else:
                remove_camera_target(active_object)
        return {'FINISHED'}


class AddLensDriver(bpy.types.Operator):
    bl_idname = "scene.add_lens_driver"
    bl_label = "Add Lens Driver"

    func: bpy.props.StringProperty()

    def add_custom_property(self, camera):
        # 定义自定义属性
        bpy.types.Camera.lens_param = bpy.props.FloatProperty(
            name="lens_param",
            default=1.0,
            min=1.0,
        )
        # 设置自定义属性
        camera["lens_param"] = 1.0

    def add_focal_length_driver(self, context, selected_camera):
        # 检查相机目标
        targets_obj = find_obj(selected_camera.name + "_Alfred_target")
        if targets_obj is None:
            targets_obj = create_empty_at_camera_position(context, selected_camera, "_Alfred_target", "PLAIN_AXES")
            context.view_layer.objects.active = selected_camera
        # 获取相机的数据对象
        camera_data = selected_camera.data
        lens = camera_data.lens
        self.add_custom_property(camera_data)
        distance = self.calculate_distance(selected_camera, targets_obj)
        if distance > 0:
            selected_camera.data["lens_param"] = lens / distance
        # 添加焦距驱动器
        target_fcurve = camera_data.driver_add("lens")
        driver = target_fcurve.driver
        driver.type = 'SCRIPTED'

        # 设置驱动器表达式
        driver.expression = "var * Lens_value"
        # 添加驱动器变量
        var = driver.variables.new()
        var.name = "var"
        var.type = 'LOC_DIFF'
        var.targets[0].id = selected_camera
        var.targets[1].id = targets_obj

        Lens_value = driver.variables.new()
        Lens_value.name = "Lens_value"
        Lens_value.type = 'SINGLE_PROP'
        Lens_value.targets[0].id_type = "CAMERA"
        Lens_value.targets[0].id = selected_camera.data
        Lens_value.targets[0].data_path = '["lens_param"]'

    def calculate_distance(self, obj1, obj2):
        # 获取两个物体的世界坐标
        loc_obj1 = obj1.matrix_world.translation
        loc_obj2 = obj2.matrix_world.translation
        # 计算两个物体之间的距离
        distance = (loc_obj1 - loc_obj2).length
        return distance

    def execute(self, context):
        # 获取当前选择的相机
        selected_camera = bpy.context.active_object
        if selected_camera.type == 'CAMERA':
            # self.has_driver(selected_camera)
            if self.func == "add":
                self.add_focal_length_driver(context, selected_camera)
            else:
                if selected_camera.data.get("lens_param"):
                    del selected_camera.data["lens_param"]
                # 删除相机的焦距驱动
                cam_data = selected_camera.data
                lens = cam_data.lens
                if cam_data.animation_data and cam_data.animation_data.drivers:
                    for driver in cam_data.animation_data.drivers:
                        if driver.data_path == "lens":
                            cam_data.driver_remove("lens")
                            cam_data.lens = lens
                Alfred_target = find_obj(selected_camera.name + "_Alfred_target")
                if Alfred_target:
                    bpy.data.objects.remove(Alfred_target, do_unlink=True)

        return {"FINISHED"}


# 为相机添加标记
class AddCameraMarker(bpy.types.Operator):
    bl_idname = "scene.add_camera_marker"
    bl_label = "Add Camera Marker"
    #    bl_options = {'REGISTER', 'UNDO'}

    idx: bpy.props.IntProperty()
    name: bpy.props.StringProperty()

    def show_status_message(self, context):
        self.layout.label(text="操作已完成！", icon='INFO')

    def execute(self, context):
        context.scene.camera_list_active_index = self.idx
        camera_obj = find_camera_by_name(self.name)
        if camera_obj is None:
            set_status_bar_message("相机丢失")
            return {'FINISHED'}
        current_frame = context.scene.frame_current
        active_marker = None
        for marker in context.scene.timeline_markers:
            if current_frame == marker.frame:
                active_marker = marker
                active_marker.frame = current_frame
                active_marker.camera = camera_obj
                break
        if active_marker is None:
            active_marker = context.scene.timeline_markers.new("")
            active_marker.frame = current_frame
            active_marker.camera = camera_obj
        #        active_marker.select = True

        return {'FINISHED'}


# 在当前视口创建相机
class VIEW3D_OT_align_view_to_camera(bpy.types.Operator):
    bl_idname = "scene.align_view_to_camera"
    bl_label = "Align View to Camera"

    # bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        current_scene_name = bpy.context.scene.name
        bpy.context.window.scene = bpy.data.scenes[current_scene_name]
        if context.space_data.region_3d.view_perspective != 'CAMERA':
            # 新建相机对象
            bpy.ops.object.camera_add()

            # 获取新建的相机对象
            active_camera = bpy.context.active_object

            # 设置新建的相机为活动相机
            bpy.context.scene.camera = active_camera

            # 执行对准视图到选中的相机
            bpy.ops.view3d.camera_to_view()
            bpy.ops.view3d.view_center_camera()

            add_camera(context)
            self.report({'INFO'}, '创建完成{}'.format(active_camera.name))

        return {'FINISHED'}


# 记录输出路径
class CameraFileOperator(bpy.types.Operator):
    bl_idname = "scene.camera_file"
    bl_label = "Camera File"

    def execute(self, context):
        camera_list=[obj for obj in bpy.context.selected_objects if obj.type == 'CAMERA']
        filepath = bpy.context.scene.render.filepath
        for active_camera in camera_list:
            for item in context.scene.camera_list:
                if item.name_camera == active_camera.name:
                    item.filepath = filepath
                    active_camera["filepath"] = filepath
                    item.name = f"{active_camera.get('world')}_{active_camera.name}_{active_camera.get('resolution_x')}*{active_camera.get('resolution_y')}:{active_camera.get('filepath')}"
                    # 添加自定义属性

                    self.report({'INFO'}, '路径已记录{}'.format(filepath))
        return {'FINISHED'}


# 找到相机
class SelectCameraOperator(bpy.types.Operator):
    bl_idname = "scene.set_active_camera"
    bl_label = "Select Camera"

    idx: bpy.props.IntProperty()

    def get_end_frame(self, context, cur_marker):

        markers = self.short_markers(context)
        marker_index = markers.index(cur_marker[-1])

        # 获取当前标记的起始帧
        start_frame = cur_marker[-1].frame
        # 获取下一个标记（如果有的话）
        if markers[-1].frame == start_frame:
            return context.scene.frame_end
        else:
            next_marker = markers[marker_index + 1]
            return next_marker.frame - 1

    def get_marker(self, context, camera_obj):
        marker_list=[]
        for marker in context.scene.timeline_markers:
            if marker.camera and marker.camera.name == camera_obj.name:
                marker_list.append(marker)
        return marker_list

    def short_markers(self, context):
        markers = context.scene.timeline_markers
        sorted_markers = sorted(markers, key=lambda m: m.frame)
        return sorted_markers

    def set_preview(self, context, marker_list):
        # 设置预览帧
        frame_start = marker_list[0].frame
        frame_end = self.get_end_frame(context, marker_list)
        context.scene.use_preview_range = True
        context.scene.frame_preview_start = frame_start
        context.scene.frame_preview_end = frame_end

    def set_render(self, context, marker_list):
        # 设置预览帧
        frame_start = marker_list[0].frame
        frame_end = self.get_end_frame(context, marker_list)
        context.scene.frame_start = frame_start
        context.scene.frame_end = frame_end

    def find_collection_by_name(self, view_layer, collection_name):
        for layer_collection in view_layer.layer_collection.children:
            if layer_collection.collection.name == collection_name:
                return layer_collection
        return None


    def execute(self, context):
        context.scene.camera_list_active_index = self.idx
        active_camera = context.scene.camera_list[self.idx]

        switch_collections(active_camera)
        camera_obj = find_camera_by_name(active_camera.name_camera)
        if camera_obj:
            c = get_object_collection_name(camera_obj)
            if c and c != "Scene Collection":
                set_collection_exclude_in_view_layer(c, False)
            if active_camera.world:
                context.scene.world = active_camera.world

            bpy.ops.object.select_all(action='DESELECT')
            current_scene_name = bpy.context.scene.name
            bpy.context.window.scene = bpy.data.scenes[current_scene_name]
            context.view_layer.objects.active = camera_obj
            camera_obj.select_set(True)
            bpy.ops.view3d.object_as_camera()
            bpy.ops.view3d.view_selected()
            bpy.ops.view3d.view_camera()
            if bpy.ops.view3d.view_center_camera.poll():
                bpy.ops.view3d.view_center_camera()
            else:
                self.report({'INFO'}, '画面可能隐藏了相机{}或者锁定相机画面！'.format(camera_obj.name))
            if camera_obj.get("resolution_x") and camera_obj.get("resolution_y"):
                context.scene.render.resolution_x = camera_obj["resolution_x"]
                context.scene.render.resolution_y = camera_obj["resolution_y"]
            if camera_obj.get('filepath'):
                context.scene.render.filepath = camera_obj['filepath']
            if context.scene.check_preview is True:
                marker = self.get_marker(context, camera_obj)
                if marker:
                    self.set_preview(context, marker)
            if context.scene.check_render is True:
                marker = self.get_marker(context, camera_obj)
                if marker:
                    self.set_render(context, marker)
        else:
            self.report({'INFO'}, '相机已经丢失'.format(active_camera.name_camera))
        return {'FINISHED'}


# 动态更新
class UpdateSwitchOperator(bpy.types.Operator):
    bl_idname = "scene.update_switch_op"
    bl_label = "Update Switch Operator"

    def execute(self, context):
        if bpy.types.Scene.update_switch:
            bpy.types.Scene.update_switch = False
            if bpy.app.handlers.frame_change_pre:
                unreg_handlers()
                self.report({'INFO'}, '已关闭动态更新')

        else:
            bpy.types.Scene.update_switch = True
            reg_handlers()
            self.report({'INFO'}, '已开启动态更新')

        return {'FINISHED'}


# 添加相机
class AddCameraOperator(bpy.types.Operator):
    bl_idname = "scene.add_camera"
    bl_label = "Add Camera"

    def execute(self, context):
        # 调用 add_camera 方法
        add_camera(context)
        self.report({'INFO'}, "更新完成")
        return {'FINISHED'}



# 读取
class ReadCameraOperator(bpy.types.Operator):
    bl_idname = "scene.read_camera"
    bl_label = "Read Camera"

    def execute(self, context):
        # 调用 read_camera 方法
        self.read_camera(context)
        self.report({'INFO'}, "读取完成")
        return {'FINISHED'}
    def find_in_camera_list(self,context,active_camera):
        for item in context.scene.camera_list:
            if item.name_camera == active_camera.name:
                return True
        return False

    def read_camera(self, context):
        cameras = [obj for obj in context.selected_objects if obj.type == 'CAMERA']
        for active_camera in cameras:
            if self.find_in_camera_list(context,active_camera):
                continue
            if active_camera.get('resolution_x') and active_camera.get('resolution_y'):
                active_item = context.scene.camera_list.add()
                active_item.name_camera = active_camera.name
                active_item.resolution_x = int(active_camera.get('resolution_x'))
                active_item.resolution_y = int(active_camera.get('resolution_y'))
                if active_camera.get('filepath'):
                    active_item.filepath = active_camera.get('filepath')
                if active_camera.get('world'):
                    world = bpy.data.worlds.get(active_camera.get('world'))
                    if world:
                        active_item.world = world
                active_item.name = f"{active_camera.get('world')}_{active_camera.name}_{active_camera.get('resolution_x')}*{active_camera.get('resolution_y')}:{active_camera.get('filepath')}"
            bpy.context.view_layer.objects.active = active_camera


# 移除
class RemoveCameraOperator(bpy.types.Operator):
    bl_idname = "scene.remove_camera"
    bl_label = "Remove Camera"
    idx: bpy.props.IntProperty()

    def execute(self, context):
        active_item = context.scene.camera_list[self.idx]
        active_camera = find_camera_by_name(active_item.name_camera)
        # 删除列表中的相机
        context.scene.camera_list.remove(self.idx)
        context.scene.camera_list_active_index = len(context.scene.camera_list) - 1
        # 删除自定义
        if active_camera:
            # 删除相机对象的自定义属性
            if "resolution_x" in active_camera:
                del active_camera["resolution_x"]
            if "resolution_y" in active_camera:
                del active_camera["resolution_y"]
            if "filepath" in active_camera:
                del active_camera["filepath"]
            if "world" in active_camera:
                del active_camera["world"]
        # 删除目标
        if context.scene.reverse_order and active_camera:
            Alfred_target = find_obj(active_camera.name + "_Alfred_target")
            target = find_obj(active_camera.name + "_target")
            if Alfred_target:
                bpy.data.objects.remove(Alfred_target, do_unlink=True)
            if target:
                bpy.data.objects.remove(target, do_unlink=True)
            bpy.data.objects.remove(active_camera)

        return {'FINISHED'}


classes = [
    CollectionItem,
    AddCameraMarker,
    CameraTargetOperator,
    AddLensDriver,
    VIEW3D_OT_align_view_to_camera,
    CameraFileOperator,
    SelectCameraOperator,
    CameraExtraPropGroup,
    CameraExtraPanel,
    AddCameraOperator,
    ReadCameraOperator,
    RemoveCameraOperator,
    UpdateSwitchOperator,
    CameraCollection,

]


def register():
    fetch()
    for cls in classes:
        bpy.utils.register_class(cls)

    bpy.types.Scene.camera_list = bpy.props.CollectionProperty(type=CameraExtraPropGroup)
    bpy.types.Scene.camera_list_active_index = bpy.props.IntProperty(default=0)
    bpy.types.Scene.update_switch = False
    bpy.types.Scene.check_preview = bpy.props.BoolProperty(default=False)
    bpy.types.Scene.check_render = bpy.props.BoolProperty(default=False)

    for item in bpy.app.handlers.frame_change_pre:
        bpy.app.handlers.frame_change_pre.remove(item)



def unregister():
    for cls in classes:
        bpy.utils.unregister_class(cls)

    del bpy.types.Scene.camera_list
    del bpy.types.Scene.camera_list_active_index
    del bpy.types.Scene.update_switch
    for item in bpy.app.handlers.frame_change_pre:
        bpy.app.handlers.frame_change_pre.remove(item)


if __name__ == "__main__":
    register()

