bl_info = {  
    "name": "REDHALO_IMPORT_MAX_SCENE",
    "author": "Red Halo Studio",
    "version": (0, 9, 5),
    "blender": (2, 80, 0),
    "location": "View 3D > Tools > Red Halo Tools",
    "description": "导入3ds Max场景",
    "wiki_url": "",
    "tracker_url": "",
    "category": "Import"
 }

from math import pi, radians
import bpy, bpy_extras, tempfile, os, traceback, bpy_extras.image_utils, subprocess # type: ignore
from bpy.types import Operator, PropertyGroup, AddonPreferences  # type: ignore
from bpy.props import StringProperty, BoolProperty, EnumProperty # type: ignore
from mathutils import Matrix, Vector  # type: ignore
from xml.etree import ElementTree as xml
from bpy_extras.io_utils import ImportHelper #poll_file_object_drop,  # type: ignore
from struct import *

__version__ = "0.9.5"

## 检测Max文件版本号 
def FindMax(filepath):
    version = ""
    sv = ""
    max_version = {2009: 10, 2010: 11, 2010:12, 2011:13, 2012:14, 2013:15, 2014:16, 2015:17, 2016:18, 2017:19, 2018:20, 2019:21, 2020:22, 2021:23, 2022:24, 2023:25, 2024:26, 2025:27, 2026:28, 2027:29, 2028:30}
    vr_v = 10000
    with open(filepath, "rb") as f:
        buff = f.read()
        # Save Max Version
        save_bit = (b"\x33\x64\x73\x20\x4D\x61\x78\x20\x56\x65\x72\x73\x69\x6F\x6E\x3A", b"\x33\x00\x64\x00\x73\x00\x20\x00\x4D\x00\x61\x00\x78\x00\x20\x00\x56\x00\x65\x00\x72\x00\x73\x00\x69\x00\x6F\x00\x6E\x00\x3A\x00", b"\x33\x64\x73\x20\x4D\x61\x78\x20\xE7\x89\x88\xE6\x9C\xAC\x3A\x20")
        save_bit_offset = (36, 48, 24)
        for s in zip(save_bit, save_bit_offset):
            fi = buff.find(s[0])
            if fi != -1:
                f.seek(fi)
                version = f.read(s[1]).decode("utf-8", "ignore").replace("\x00","").replace(",", ".").replace(" ","").replace("3dsMaxVersion:", "").replace("3dsMax版本:", "")

        # Save As Version
        SaveAs_bit = (b"\x53\x61\x76\x65\x64\x20\x41\x73", b"\x53\x00\x61\x00\x76\x00\x65\x00\x64\x00\x20\x00\x41\x00\x73\x00\x20", b"\xE5\x8F\xA6\xE5\xAD\x98\xE4\xB8\xBA\xE7\x89\x88\xE6\x9C\xAC\x3A")
        
        # Vrend sting : vrender版本号.dlr
        vrend = (b"x\76x\00x\72x\00x\65x\00x\6Ex\00x\64x\00x\65x\00x\72x\00")

        SaveAs_bit_offset = (28, 48, 24)        
        for s in zip(SaveAs_bit, SaveAs_bit_offset):
            fi = buff.find(s[0])
            if fi != -1:
                f.seek(fi)
                sv = f.read(s[1]).decode("utf-8","ignore").replace("\x00","").replace(",", ".").replace(" ", "").replace("SavedAsVersion:", "").replace("另存为版本:", "")
            else:
                sv = version
        # Vrend sting : vrender版本号.dlr
        vrend = (b"\x76\x00\x72\x00\x65\x00\x6E\x00\x64\x00\x65\x00\x72\x00")
        findIndex = buff.find(vrend)
        if  findIndex != -1: 
            f.seek(findIndex)
            v_version = f.read(32).decode("utf-16")
            vrender_version = int(os.path.splitext(v_version)[0].replace("vrender", ""))
            vr_v = max_version[vrender_version]

    mv = 10000 if version == "" else int(version.split(".")[0])
    av = 10000 if sv == "" else int(sv.split(".")[0])
    vv = min(mv, av, vr_v)
    return vv

def GetImportPath():
    # Reg Name in Windows
    RegRoot = ConnectRegistry(None, HKEY_CURRENT_USER)
    RegSubDir = "SOFTWARE\\REDHALO"

    try:
        RegKey = OpenKey(RegRoot, RegSubDir)
        keyName = EnumValue(RegKey, 0)
        return keyName[1]  
    except:
        return ""          

def Create_ColorCorrection():
    # Color Correction Node
    CC_group = bpy.data.node_groups.new('ColorCorrection_RH', 'ShaderNodeTree')
    group_inputs = CC_group.nodes.new('NodeGroupInput')
    group_inputs.location = (0, 0)
    items =[]
    if bpy.app.version < (4, 00, 0):        
        CC_group.inputs.new('NodeSocketColor','Color')
        CC_group.inputs.new('NodeSocketFloat','Bright')
        CC_group.inputs.new('NodeSocketFloat','Contrast')
        CC_group.inputs.new('NodeSocketFloat','Hue')
        CC_group.inputs.new('NodeSocketFloat','Saturation')
        CC_group.inputs.new('NodeSocketFloat','Invert')
        CC_group.inputs.new('NodeSocketFloat','Gamma')
        items = CC_group.inputs
    else:
        CC_group.interface.new_socket(name="Color", in_out = "INPUT", socket_type="NodeSocketColor")
        CC_group.interface.new_socket(name="Bright", in_out = "INPUT", socket_type="NodeSocketFloat")
        CC_group.interface.new_socket(name="Contrast", in_out = "INPUT", socket_type="NodeSocketFloat")
        CC_group.interface.new_socket(name="Hue", in_out = "INPUT", socket_type="NodeSocketFloat")
        CC_group.interface.new_socket(name="Saturation", in_out = "INPUT", socket_type="NodeSocketFloat")
        CC_group.interface.new_socket(name="Invert", in_out = "INPUT", socket_type="NodeSocketFloat")
        CC_group.interface.new_socket(name="Gamma", in_out = "INPUT", socket_type="NodeSocketFloat")
        items = CC_group.interface.items_tree

    items["Color"].default_value = (1,1,1,1)

    items["Bright"].min_value = -100
    items["Bright"].max_value = 100

    items["Contrast"].min_value = -100
    items["Contrast"].max_value = 100

    items["Hue"].default_value = 0.5
    items["Hue"].min_value = 0
    items["Hue"].max_value = 1

    items["Saturation"].default_value = 1
    items["Saturation"].min_value = 0
    items["Saturation"].max_value = 1

    items["Invert"].default_value = 0
    items["Invert"].min_value = 0
    items["Invert"].max_value = 1

    items["Gamma"].default_value = 1
    items["Gamma"].min_value = 0.01
    items["Gamma"].max_value = 10

    group_outputs = CC_group.nodes.new("NodeGroupOutput")
    group_outputs.location = (1000, 0)
    if bpy.app.version < (4, 00, 0):        
        CC_group.outputs.new("NodeSocketColor", "Color")
    else:
        CC_group.interface.new_socket(name="Color", in_out = "OUTPUT", socket_type="NodeSocketColor")

    BCNode = CC_group.nodes.new("ShaderNodeBrightContrast")
    BCNode.location = (200, 0)
    HSVNode = CC_group.nodes.new("ShaderNodeHueSaturation")
    HSVNode.location = (400, 0)
    InvertNode =  CC_group.nodes.new("ShaderNodeInvert")
    InvertNode.location = (600, 0)
    GammaNode = CC_group.nodes.new("ShaderNodeGamma")
    GammaNode.location = (800, 0)

    CC_group.links.new(group_inputs.outputs["Color"], BCNode.inputs["Color"])
    CC_group.links.new(group_inputs.outputs["Bright"], BCNode.inputs["Bright"])
    CC_group.links.new(group_inputs.outputs["Contrast"], BCNode.inputs["Contrast"])
    CC_group.links.new(group_inputs.outputs["Hue"], HSVNode.inputs["Hue"])
    CC_group.links.new(group_inputs.outputs["Saturation"], HSVNode.inputs["Saturation"])
    CC_group.links.new(group_inputs.outputs["Invert"], InvertNode.inputs["Fac"])
    CC_group.links.new(group_inputs.outputs["Gamma"], GammaNode.inputs["Gamma"])

    CC_group.links.new(BCNode.outputs["Color"], HSVNode.inputs["Color"])
    CC_group.links.new(HSVNode.outputs["Color"], InvertNode.inputs["Color"])
    CC_group.links.new(InvertNode.outputs["Color"], GammaNode.inputs["Color"])
    CC_group.links.new(GammaNode.outputs["Color"], group_outputs.inputs["Color"])

def ConvertMatrix(matrixstr):
    ma = matrixstr.split(",")
    matrix = Matrix()

    for i in range(4):
        for j in range(4):
            index = i*4+j
            matrix[i][j] = float(ma[index])
    
    return matrix

def ConvertColor(color):
    #_c = (color[7:-1]+" 255").split()
    _c = color.split(',')
    _clr = [float(c) for c in _c]
    return _clr

def ConvertColorMax(color):
    _c = (color[7:-1]+" 255").split()
    _clr = [float(c)/255.0 for c in _c]
    return _clr

def CollectImagesNode(node):
    allnodes = []
    allnodes.append(node)    
    inputs = [n for n in node.inputs if n.is_linked]
    
    for input in inputs:
        for l in input.links:
            allnodes.extend(CollectImagesNode(l.from_node))
            
    return allnodes

def CreateNode(nodes, links, xmlPath, ParentNode, texmap_type):
    '''
    #### nodes      :节点
    #### links      :连接
    #### xmlPath    :查找路径
    #### ParentNode :父级节点
    #### type       :节点类型
    '''
    texmap = xmlPath    
    _node = texmap.findall("./")
    
    # Bitmap 已完成
    if texmap_type in ("Bitmap", "VRayBitmap") :
        bitmap_path = _node[0].text
        if bitmap_path != "":            
            # Scale
            uTile = float(texmap.attrib["u_tiling"])
            vTile = float(texmap.attrib["v_tiling"])
            # Location
            uOffset = float(texmap.attrib["u_offset"])
            vOffset = float(texmap.attrib["v_offset"])
            #Rotation
            wAngle = float(texmap.attrib["w_angle"])
            vAngle = float(texmap.attrib["v_angle"])
            uAngle = float(texmap.attrib["u_angle"])
            # Mapping
            mapping = int(texmap.attrib["mapping"])
            mappingtype = int(texmap.attrib["mappingtype"])
            
            ImageShader = CreateImageNode(nodes, links, bitmap_path, position=(uOffset, vOffset, 0), rotation=(radians(uAngle), radians(vAngle), radians(wAngle)), scale=(uTile, vTile, 1), mapping=mapping, mappingtype=mappingtype)     

            if ImageShader:
                links.new(ImageShader.outputs["Color"], ParentNode)
    
    # 平铺 部分完成
    elif texmap_type == "Tiles" :
        BrickShader = nodes.new("ShaderNodeTexBrick")
        # BrickShader.location = p_loc - Vector((p_width + 100, 0))
        links.new(BrickShader.outputs["Color"], ParentNode)

        # 平铺纹理基础参数设置
        Brick_Color = texmap.attrib["brick_color"]
        Mortar_Color = texmap.attrib["mortar_color"]
        BrickShader.inputs["Color1"].default_value = ConvertColorMax(Brick_Color)
        BrickShader.inputs["Color2"].default_value = ConvertColorMax(Brick_Color)
        BrickShader.inputs["Mortar"].default_value = ConvertColorMax(Mortar_Color)

        BrickShader.inputs["Scale"].default_value = 1
        BrickShader.inputs["Mortar Size"].default_value = float(texmap.attrib["horizontal_gap"]) / 100
        BrickShader.inputs["Brick Width"].default_value = 1 / float(texmap.attrib["horizontal_count"])
        BrickShader.inputs["Row Height"].default_value = 1 / float(texmap.attrib["vertical_count"])

        Brick_Type = texmap.attrib["tile_type"]
        Brick_offset = 0.5
        if Brick_Type == "5" :
            Brick_offset = 0
        elif Brick_Type == "0" :
            Brick_offset = texmap.attrib["line_shift"]
        
        BrickShader.offset = Brick_offset
        
        CreateUVMapping(nodes, links, BrickShader)

        # 判断分缝线有没有纹理，如果有创建相关节点
        if texmap.attrib["mortar_map"] != "undefined" :
            MortarNode = _node[0][0]
            MortarType = texmap.attrib["mortar_map"].split(":")[-1]
            CreateNode(nodes, links, MortarNode, BrickShader.inputs["Mortar"], MortarType)
        
        # 判断平铺有没有纹理，如果有创建相关节点
        if texmap.attrib["bricks_map"] != "undefined" :
            TileNode = _node[1][0]
            TileType = texmap.attrib["bricks_map"].split(":")[-1]
            CreateNode(nodes, links, TileNode, BrickShader.inputs["Color1"], TileType)
            CreateNode(nodes, links, TileNode, BrickShader.inputs["Color2"], TileType)

    # 混合 已完成
    elif texmap_type in ("Mix", "RGB_Multiply", "CoronaMix", "RGB_Multiply", "VRayCompTex") :
        #  #### Params : Color1 Color2 MixAmount Texmap1 Texmap2 TexmapMask MixMode
        #  MixMode:
        #  ADD SUBTRACT MULTIPLY DIVIDE MINIMUM MAXIMUM MIX GAMMA DIFFERENCE SCREEN OVERLAY COLORDODGE COLORBURN LINEARBURN LINEARLIGHT SOFTLIGHT HARDLIGHT VIVIDLIGHT PINLIGHT HARDMIX EXCLUSION
        blendMode = ('MIX', 'DARKEN', 'MULTIPLY', 'BURN', 'LIGHTEN', 'SCREEN', 'DODGE', 'ADD', 'OVERLAY', 'SOFT_LIGHT', 'LINEAR_LIGHT', 'DIFFERENCE', 'EXCLUSION', 'SUBTRACT', 'DIVIDE', 'HUE', 'SATURATION', 'COLOR', 'VALUE')
        # 旧版MixRGB
        MixShader = nodes.new("ShaderNodeMixRGB")
        
        links.new(MixShader.outputs["Color"], ParentNode)

        # 基础参数设置
        amount = texmap.attrib["mix_amount"]
        color1 = texmap.attrib["color1"]
        color2 = texmap.attrib["color2"]
        MixShader.inputs["Fac"].default_value = float(amount)/100
        MixShader.inputs["Color1"].default_value = ConvertColorMax(color1)
        MixShader.inputs["Color2"].default_value = ConvertColorMax(color2)

        MixShader.blend_type = texmap.attrib["mix_mode"] if texmap.attrib["mix_mode"] in blendMode else "MIX"

        if texmap.attrib["texmap1"] != "undefined":
            map1Node = _node[0][0]
            map1Type = texmap.attrib["texmap1"].split(":")[-1]
            CreateNode(nodes, links, map1Node, MixShader.inputs["Color1"], map1Type)

        if texmap.attrib["texmap2"] != "undefined":
            map2Node = _node[1][0]
            map2Type = texmap.attrib["texmap2"].split(":")[-1]
            CreateNode(nodes, links, map2Node, MixShader.inputs["Color2"], map2Type)

        if texmap.attrib["texmap_mask"] != "undefined":
            maskNode = _node[2][0]
            MaskType = texmap.attrib["texmap_mask"].split(":")[-1]
            CreateNode(nodes, links, maskNode, MixShader.inputs["Fac"], MaskType)

    # 棋盘格  已完成
    elif texmap_type == "Checker":
        CheckerShader = nodes.new("ShaderNodeTexChecker")
        links.new(CheckerShader.outputs["Color"], ParentNode)

        #基础参数
        color1 = texmap.attrib["color1"]
        color2 = texmap.attrib["color2"]
        map1 = texmap.attrib["map1"]
        map2 = texmap.attrib["map2"]

        CheckerShader.inputs["Color1"].default_value = ConvertColorMax(color1)
        CheckerShader.inputs["Color2"].default_value = ConvertColorMax(color2)
        CheckerShader.inputs["Scale"].default_value = 2

        if map1 != "undefined" :
            map1Node = _node[0][0]
            map1Type = map1.split(":")[-1]
            CreateNode(nodes, links, map1Node, CheckerShader.inputs["Color1"], map1Type)

        if map2 != "undefined" :
            map2Node = _node[1][0]
            map2Type = map2.split(":")[-1]
            CreateNode(nodes, links, map2Node, CheckerShader.inputs["Color2"], map2Type)
        
        CreateUVMapping(nodes, links, CheckerShader)

    # 衰减  完成IOR部分，其它类型还没找到相应实现方式
    elif texmap_type == "Falloff":        
        # 旧版MixRGB
        MixRGBShader = nodes.new("ShaderNodeMixRGB")
        links.new(MixRGBShader.outputs["Color"], ParentNode)

        color1 = texmap.attrib["color1"]
        color2 = texmap.attrib["color2"]
        MixRGBShader.inputs["Color1"].default_value = ConvertColorMax(color1)
        MixRGBShader.inputs["Color2"].default_value = ConvertColorMax(color2)

        # Map1
        if texmap.attrib["map1"] != "undefined":
            map1Node = _node[0][0]
            map1Type = texmap.attrib["map1"].split(":")[-1]
            CreateNode(nodes, links, map1Node, MixRGBShader.inputs["Color1"], map1Type)
        # Map2
        if texmap.attrib["map2"] != "undefined":
            map2Node = _node[1][0]
            map2Type = texmap.attrib["map2"].split(":")[-1]
            CreateNode(nodes, links, map2Node, MixRGBShader.inputs["Color2"], map2Type)

        Falloff_Type = texmap.attrib["type"]
        if Falloff_Type == "2":
            FresnelShader = nodes.new("ShaderNodeFresnel")
            links.new(FresnelShader.outputs["Fac"], MixRGBShader.inputs["Fac"])
            FresnelShader.inputs["IOR"].default_value = float(texmap.attrib["ior"])
        else:
            LayerWeightNode = nodes.new("ShaderNodeLayerWeight")
            links.new(LayerWeightNode.outputs["Facing"], MixRGBShader.inputs["Fac"])

    # 颜色校正节点 已完成
    elif texmap_type in ("Color Correction", "CoronaColorCorrect"):
        # ### Params:
        # #### Color Map Hue Saturation Brightness Contrast Gamma ColorMode
        ###### ColorMode --> NORAML INVERT MONO
        cc_group = nodes.new("ShaderNodeGroup")
        # cc_group.location = p_loc - Vector((p_width + 100, 0))
        cc_group.node_tree = bpy.data.node_groups['ColorCorrection_RH']
        links.new(cc_group.outputs["Color"], ParentNode)

        cc_group.inputs["Color"].default_value = ConvertColorMax(texmap.attrib["color"])
        
        if texmap.attrib["color_mode"] == "MONO":
            cc_group.inputs["Saturation"].default_value = 0
        elif texmap.attrib["color_mode"] == "INVERT":
            cc_group.inputs["Invert"].default_value = 1

        cc_group.inputs["Bright"].default_value = float(texmap.attrib["brightness"])/100.0
        cc_group.inputs["Contrast"].default_value = float(texmap.attrib["contrast"])/100.0
        cc_group.inputs["Gamma"].default_value = 1 / float(texmap.attrib["gamma"])

        cc_group.inputs["Hue"].default_value = float(texmap.attrib["hue"])/360.0 + 0.5
        cc_group.inputs["Saturation"].default_value = float(texmap.attrib["saturation"])/100.0 + 1

        if texmap.attrib["map"] != "undefined" :
            mapNode = _node[0][0]
            mapType = texmap.attrib["map"].split(":")[-1]
            CreateNode(nodes, links, mapNode, cc_group.inputs["Color"], mapType)

    # 渐变节点,Blender不支持贴图，所以只保留颜色
    elif texmap_type == "Gradient" :
        GradientNode = nodes.new("ShaderNodeTexGradient")
        ColorRampNode = nodes.new("ShaderNodeValToRGB")

        links.new(GradientNode.outputs["Fac"], ColorRampNode.inputs["Fac"])
        links.new(ColorRampNode.outputs["Color"], ParentNode)
        
        if texmap.attrib["gradient_type"] == "1":
            GradientNode.gradient_type = "SPHERICAL"
            CreateUVMapping(nodes, links, GradientNode, position=(-0.5, -0.5, -0.86))
        else:
            CreateUVMapping(nodes, links, GradientNode, position=(-0.5, -0.5, 0))

        ColorRampNode.color_ramp.elements.new(float(texmap.attrib["color2Pos"]))

        ColorRampNode.color_ramp.elements[0].color = ConvertColorMax(texmap.attrib["color1"])
        ColorRampNode.color_ramp.elements[1].color = ConvertColorMax(texmap.attrib["color2"])
        ColorRampNode.color_ramp.elements[2].color = ConvertColorMax(texmap.attrib["color3"])        

    # 渐变坡度节点
    elif texmap_type == "Gradient Ramp" :
        pass
        GradientNode = nodes.new("ShaderNodeTexGradient")
        ColorRampNode = nodes.new("ShaderNodeValToRGB")

        links.new(GradientNode.outputs["Fac"], ColorRampNode.inputs["Fac"])
        links.new(ColorRampNode.outputs["Color"], ParentNode)

    # AO
    elif texmap_type in ("VRayDirt", "CoronaAO") : 
        # Params Radius OccludedColor UnoccludedColor Subdivs OnlySameObject TexmapRadius TexmapOccluded TexmapUnoccluded Mode
        # Mode : OUTSIDE INSIDE BOTH
        MixShader = nodes.new("ShaderNodeMixRGB")

        AONode = nodes.new("ShaderNodeAmbientOcclusion")
        AONode.location = MixShader.location - Vector((MixShader.width + 100, -100))

        links.new(AONode.outputs["AO"], MixShader.inputs["Fac"])
        links.new(MixShader.outputs["Color"], ParentNode)

        AONode.inputs["Color"].default_value = ConvertColorMax(texmap.attrib["occluded_color"])
        AONode.samples = int(texmap.attrib["subdivs"])
        AONode.inputs["Distance"].default_value = float(texmap.attrib["radius"])

        MixShader.inputs["Color1"].default_value = (0, 0, 0, 1)
        MixShader.inputs["Color2"].default_value = (1, 1, 1, 1)

        if texmap.attrib["only_sameobject"] == "false" :
            AONode.only_local = False
        else:
            AONode.only_local = True
        
        if texmap.attrib["mode"] == "INSIDE":
            AONode.inside = True
        else:
            AONode.inside = False

        if texmap.attrib["texmap_occluded"] != "undefined":
            OccludedMap = _node[0][0]
            OccludedType = texmap.attrib["texmap_occluded"].split(":")[-1]
            CreateNode(nodes, links, OccludedMap, MixShader.inputs["Color1"], OccludedType)
        
        if texmap.attrib["texmap_unoccluded"] != "undefined":
            UnoccludedMap = _node[1][0]
            UnoccludedType = texmap.attrib["texmap_unoccluded"].split(":")[-1]
            CreateNode(nodes, links, UnoccludedMap, MixShader.inputs["Color2"], UnoccludedType)

        if texmap.attrib["texmap_radius"] != "undefined" :
            RadiusMap = _node[2][0]
            RadiusType = texmap.attrib["texmap_radius"].split(":")[-1]
            CreateNode(nodes, links, RadiusMap, AONode.inputs[1], RadiusType)

    # 法线/凹凸
    elif texmap_type in ("VRayColor2Bump", "VRayNormalMap", "CoronaNormal", "VRayBump2Normal", "CoronaBumpConverter") :
        '''
        #### Params : NormalMap BumpMap BumpStrength NormalStrength FlipRed FilpGreen SweepRedGreen
        '''
        HasBump = True if texmap.attrib["bump_map"] != "undefined" else False
        HasNormal = True if texmap.attrib["normal_map"] != "undefined" else False

        if HasNormal :
            NormalNode = nodes.new("ShaderNodeNormalMap")
            NormalNode.inputs["Strength"].default_value = 1.0 # float(texmap.attrib["NormalStrength"])

            normal_map = texmap[0][0]
            normal_map_type = texmap.attrib["normal_map"].split(":")[-1]
            CreateNode(nodes, links, normal_map, NormalNode.inputs["Color"], normal_map_type)

        if HasBump :
            BumpNode = nodes.new("ShaderNodeBump")
            BumpNode.inputs["Strength"].default_value = 0.1 #float(texmap.attrib["BumpStrength"])

            Bump_Map = texmap[1][0]
            Bump_Map_Type = texmap.attrib["bump_map"].split(":")[-1]
            CreateNode(nodes, links, Bump_Map, BumpNode.inputs["Height"], Bump_Map_Type)
        

        ### 如果有凹凸并且有法线--全
        ### 如果有凹凸无法线 --- 只连bump
        ### 如果没有凹凸只有法线  --- 直连Normal
        if HasBump:
            if HasNormal: ## +B + N
                links.new(NormalNode.outputs["Normal"], BumpNode.inputs["Normal"])
                links.new(BumpNode.outputs["Normal"], ParentNode)
            else:  ## +B - N
                links.new(BumpNode.outputs["Normal"], ParentNode)
        else:
            ## -B +N
            if HasNormal:
                links.new(NormalNode.outputs["Normal"], ParentNode)
            #else: ## -B -N
            #    links.new(BumpNode.outputs["Normal"], ParentNode)

    elif texmap_type == "Vertex Color" :
        VCNode = nodes.new("ShaderNodeAttribute")
        if bpy.app.version < (3, 4, 0):
            VCNode.attribute_name = "Col"
        else:
            VCNode.attribute_name = "Attribute"

        links.new(VCNode.outputs["Color"], ParentNode)
    
    elif texmap_type in ("CoronaFrontBack"):
        
        # 旧版MixRGB
        MixShader = nodes.new("ShaderNodeMixRGB")
        links.new(MixShader.outputs["Color"], ParentNode)

        # 基础参数设置
        color1 = texmap.attrib["front_color"]
        color2 = texmap.attrib["back_color"]
        MixShader.inputs["Color1"].default_value = ConvertColorMax(color1)
        MixShader.inputs["Color2"].default_value = ConvertColorMax(color2)

        ## Object Backing Node
        Backfacing_Node = nodes.new("ShaderNodeNewGeometry")
        links.new(Backfacing_Node.outputs["Backfacing"], MixShader.inputs["Fac"])

        if texmap.attrib["front_texmap"] != "undefined":
            map1Node = _node[0][0]
            map1Type = texmap.attrib["front_texmap"].split(":")[-1]
            CreateNode(nodes, links, map1Node, MixShader.inputs["Color1"], map1Type)

        if texmap.attrib["back_texmap"] != "undefined":
            map2Node = _node[1][0]
            map2Type = texmap.attrib["back_texmap"].split(":")[-1]
            CreateNode(nodes, links, map2Node, MixShader.inputs["Color2"], map2Type)

    elif texmap_type in ("Composite"):
        #Blender Mode
        blendMode = ('MIX', 'DARKEN', 'MULTIPLY', 'BURN', 'LIGHTEN', 'SCREEN', 'DODGE', 'ADD', 'OVERLAY', 'SOFT_LIGHT', 'LINEAR_LIGHT', 'DIFFERENCE', 'EXCLUSION', 'SUBTRACT', 'DIVIDE', 'HUE', 'SATURATION', 'COLOR', 'VALUE')        
        
        OutNode = None
        for i,layer in enumerate(texmap):
            mode = layer.attrib["mode"]

            opacity = float(layer.attrib["opacity"])
            layer_map = layer.attrib["layer"]
            layer_mask = layer.attrib["mask"]

            if layer_map != "undefined":
                map1Node = layer[0][0]
                map1Type = layer_map.split(":")[-1]

                if i == 1:
                    # 旧版MixRGB
                    MixShader = nodes.new("ShaderNodeMixRGB")
                    # First
                    layer_1= texmap[0]
                    layer_1_map = layer_1[0][0]
                    layer_1_type = layer_1.attrib["layer"].split(":")[-1]
                    CreateNode(nodes, links, layer_1_map, MixShader.inputs["Color1"], layer_1_type)
                    # Sceond
                    CreateNode(nodes, links, map1Node, MixShader.inputs["Color2"], map1Type)
                    MixShader.blend_type = mode if mode in blendMode else 'MIX'
                    MixShader.inputs["Fac"].default_value = opacity

                    if layer_mask != "undefined":
                        maskNode = layer[1][0]
                        maskType = layer_mask.split(":")[-1]
                        CreateNode(nodes, links, maskNode, MixShader.inputs["Fac"], maskType)

                    OutNode = MixShader
                elif i > 1: 
                    # 旧版MixRGB
                    MixShader = nodes.new("ShaderNodeMixRGB")
                    links.new(OutNode.outputs["Color"], MixShader.inputs["Color1"])
                    CreateNode(nodes, links, map1Node, MixShader.inputs["Color2"], map1Type)
                    MixShader.blend_type = mode if mode in blendMode else 'MIX'
                    MixShader.inputs["Fac"].default_value = opacity

                    if layer_mask != "undefined":
                        maskNode = layer[1][0]
                        maskType = layer_mask.split(":")[-1]
                        CreateNode(nodes, links, maskNode, MixShader.inputs["Fac"], maskType)

                    OutNode = MixShader
        
        links.new(OutNode.outputs["Color"], ParentNode)

def CreateUVMapping(nodes, links, ParentNode, Coords="UV", position=(0, 0, 0), rotation=(0, 0, 0), scale=(1, 1, 1)):
    '''
    #### Coords      :坐标,默认为 “UV”，参数列表:Generated, Normal, UV, Object, Camera, Window, Reflection
    #### position    :移动，默认为(0,0,0)
    #### rotation    :旋转角度
    #### scale       :缩放
    #### ParentNode  :父级节点
    '''
    mapShader = nodes.new("ShaderNodeMapping")
    links.new(mapShader.outputs["Vector"], ParentNode.inputs["Vector"])
    
    mapShader.location = ParentNode.location - Vector((ParentNode.width, 0))

    mapShader.inputs["Location"].default_value = position
    mapShader.inputs["Rotation"].default_value = rotation
    mapShader.inputs["Scale"].default_value = scale

    texcoordShader = nodes.new("ShaderNodeTexCoord")

    links.new(texcoordShader.outputs[Coords], mapShader.inputs["Vector"])

def CreateImageNode(nodes, links, bitmap, Coors="UV", position=(0, 0, 0), rotation=(0, 0, 0), scale=(1, 1, 1), mapping=0, mappingtype=0):
    '''
    #### bitmap      :图像路径
    #### Coords      :坐标,默认为 “UV”，参数列表:Generated, Normal, UV, Object, Camera, Window, Reflection
    #### position    :移动，默认为(0,0,0)
    #### rotation    :旋转角度
    #### scale       :缩放
    #### ParentNode  :父级节点
    '''

    _, filename = os.path.split(bitmap)
    img = bpy_extras.image_utils.load_image(bitmap, check_existing=True)    
    ImageShader = None
    if img:        
        if mappingtype == 1:
            # Sphere Environ
            if mapping == 0:
                ImageShader = nodes.new("ShaderNodeTexEnvironment")
        else:
            ImageShader = nodes.new("ShaderNodeTexImage")
    
        img.name = filename
        ImageShader.image = img

        if mappingtype == 1:
            # Sphere Environ
            if mapping == 0:
                CreateUVMapping(nodes, links, ImageShader, "Generated", position, rotation, scale)
            if mapping == 3:
                CreateUVMapping(nodes, links, ImageShader, "Window", position, rotation, scale)
        else:
            if position != (0, 0, 0) or rotation != (0, 0, 0) or scale != (1, 1, 1):
                CreateUVMapping(nodes, links, ImageShader, "UV", position, rotation, scale)

    return ImageShader

def CreateSingleMtl(nodes, links, xmlPath):
    '''
    ### nodes : 节点组
    ### xmlPath: 查找节点位置
    '''
    bsdfShader = nodes.new("ShaderNodeBsdfPrincipled")

    NodeName =  ("diffuse",     "metallic", "reflection",           "roughness",    "anisotropic",  "anisotropic_rotation",  "sheen",       "sheen_color",  "sheen_roughness",  "coat",         "coat_roughness",  "ior",  "refraction",            "emission",         "emission_strength",    "opacity",  "bump")
    inputName = ("Base Color",  "Metallic", "Specular IOR Level",   "Roughness",    "Anisotropic",  "Anisotropic Rotation", "Sheen Weight", "Sheen Tint",   "Sheen Roughness",  "Coat Weight",  "Coat Roughness",  "IOR",  "Transmission Weight",   "Emission Color",   "Emission Strength",    "Alpha",    "Normal")

    if bpy.app.version < (4, 0, 0):
        NodeName  = ("diffuse",    "metallic", "reflection", "roughness", "anisotropic", "anisotropic_rotation", "sheen", "coat", "coat_roughness", "ior", "refraction", "refract_roughness", "emission", "emission_strength", "opacity", "bump")
        inputName = ("Base Color", "Metallic", "Specular", "Roughness", "Anisotropic", "Anisotropic Rotation", "Sheen", "Clearcoat","Clearcoat Roughness", "IOR", "Transmission", "Transmission Roughness", "Emission", "Emission Strength", "Alpha", "Normal")

    useRoughness = xmlPath.attrib.get("use_roughness") == "true"

    for path, input in zip(NodeName, inputName):
        node = xmlPath.find("./%s" % path)

        Default_Value = 1.0
        clr = node.attrib.get("color")
        if clr:
            Default_Value = ConvertColorMax(clr)
        
        amt = node.attrib.get("amount")
        if amt :
            Default_Value = float(amt)
        
        texmap = node.attrib["texmap"]
        texmap_type = texmap.split(":")[-1]
        texmap_node = xmlPath.find("./%s/" % path)

        parent_node = bsdfShader.inputs[input]  
        if path in ("roughness", "coat_roughness", "refract_roughness", "sheen_roughness") :
            if useRoughness or path == "refract_roughness":
                parent_node.default_value = 1 - Default_Value
                if texmap_node:
                    InvertNode = nodes.new("ShaderNodeInvert")
                    links.new(InvertNode.outputs["Color"], parent_node)
                    CreateNode(nodes, links, texmap_node, InvertNode.inputs["Color"], texmap_type)
        elif path in ("bump"):
            if texmap_node:
                if texmap_node.tag in ("normal"):
                    CreateNode(nodes, links, texmap_node, parent_node, texmap_type)
                else:
                    BumpNode = nodes.new("ShaderNodeBump")
                    BumpNode.inputs["Strength"].default_value = 0.1
                    links.new(BumpNode.outputs["Normal"], parent_node)
                    CreateNode(nodes, links, texmap_node, BumpNode.inputs["Height"], texmap_type)
        else:
            parent_node.default_value = Default_Value
            if texmap_node:
                CreateNode(nodes, links, texmap_node, parent_node, texmap_type)
    
    return bsdfShader

def import_max_scene_operator(**kwargs):    
    import_path = tempfile.gettempdir()

    MTLFile = os.path.join(import_path, "RH_M2B_TEMP", "RH_M2B.xml")
    OtherFile = os.path.join(import_path, "RH_M2B_TEMP", "RHM2B_CONTENT.xml")
    FBXFile = os.path.join(import_path, "RH_M2B_TEMP", "RH_M2B.fbx")
    USDFile = os.path.join(import_path, "RH_M2B_TEMP", "RH_M2B.usd")

    ##### Importer Options #####
    import_model = kwargs["import_model"] if "import_model" in kwargs else False
    import_light = kwargs["import_light"] if "import_light" in kwargs else False
    import_proxy = kwargs["import_proxy"] if "import_proxy" in kwargs else False
    import_camera = kwargs["import_camera"] if "import_camera" in kwargs else False
    import_material = kwargs["import_material"] if "import_material" in kwargs else False
    import_settings = kwargs["import_settings"] if "import_settings" in kwargs else False
    import_environment = kwargs["import_environment"] if "import_environment" in kwargs else False
    import_animate = kwargs["import_animate"] if "import_animate" in kwargs else False

    use_collection = kwargs["use_collection"] if "use_collection" in kwargs else False
    toggle_cycles_render = kwargs["toggle_cycles_render"] if "toggle_cycles_render" in kwargs else False

    if os.path.exists(OtherFile) and (os.path.exists(FBXFile) or os.path.exists(USDFile)):
        xml_file = xml.parse(OtherFile)
        xml_root = xml_file.getroot()
        
        # Mesh Format
        MeshFormat = xml_root.find("./settings/export_format").text
        # Gamma
        gamma = float(xml_root.find("./settings/gamma").text)
        
        # 文件单位
        fac = float(xml_root.find("./settings/meters_scale").text)        
        
        print("============RED HALO Studio============")

        # 切换为Cycles渲染器
        if toggle_cycles_render :
            bpy.context.scene.render.engine = 'CYCLES'
        # 加载模型
        if import_model :
            print("========= 开始加载模型 Loading... =========")
            try:
                if MeshFormat == "USD":
                    if bpy.app.version < (4,1,0):
                        bpy.ops.wm.usd_import(filepath=USDFile)
                    else:
                        bpy.ops.wm.usd_import(filepath=USDFile, support_scene_instancing=False)
                    # bpy.ops.wm.usd_import(filepath='', check_existing=False, filter_blender=False, filter_backup=False, filter_image=False, filter_movie=False, filter_python=False, filter_font=False, filter_sound=False, filter_text=False,
                    # filter_archive=False, filter_btx=False, filter_collada=False, filter_alembic=False, filter_usd=True, filter_obj=False, filter_volume=False, filter_folder=True, filter_blenlib=False, filemode=8, relative_path=True, 
                    # display_type='DEFAULT', sort_method='', filter_glob='*.usd', scale=1.0, set_frame_range=True, import_cameras=True, import_curves=True, import_lights=True, import_materials=True, import_meshes=True, import_volumes=True, 
                    # import_shapes=True, import_subdiv=False, import_instance_proxies=True, import_visible_only=True, create_collection=False, read_mesh_uvs=True, read_mesh_colors=True, prim_path_mask='', import_guide=False, import_proxy=True, 
                    # import_render=True, import_all_materials=False, import_usd_preview=True, set_material_blend=True, light_intensity_scale=1.0, mtl_name_collision_mode='MAKE_UNIQUE', import_textures_mode='IMPORT_PACK', import_textures_dir='//textures/', tex_name_collision_mode='USE_EXISTING')
                    for b in bpy.data.objects:
                        for c in b.constraints:
                            b.constraints.remove(c)
                    
                    for c in bpy.data.objects:
                        if c.type == "CURVE":
                            c.data.bevel_depth = 0

                if MeshFormat == "FBX":
                    use_anim = True if import_animate else False
                    bpy.ops.import_scene.fbx(filepath=FBXFile, use_custom_normals=False, use_custom_props=False, use_anim=use_anim, automatic_bone_orientation=True)
                    # bpy.ops.import_scene.fbx(filepath='', directory='', filter_glob='*.fbx', files=None, ui_tab='MAIN', use_manual_orientation=False, global_scale=1.0, bake_space_transform=False, use_custom_normals=True, use_image_search=True, 
                    # use_alpha_decals=False, decal_offset=0.0, use_anim=True, anim_offset=1.0, use_subsurf=False, use_custom_props=True, use_custom_props_enum_as_string=True, ignore_leaf_bones=False, force_connect_children=False, automatic_bone_orientation=False, 
                    # primary_bone_axis='Y', secondary_bone_axis='X', use_prepost_rot=True, axis_forward='-Z', axis_up='Y')

                for obj in bpy.data.objects:
                    object_name = obj.name
                    object_xml = xml_root.find("./geometries/*/[@name='%s']" % object_name)
                    
                    if(object_xml is not None):
                        base_object = object_xml.attrib["base_object"]

                        object_Render = True if object_xml.attrib["renderable"] == "true" else False
                        object_camera = True if object_xml.attrib["visiblecamera"] == "true" else False
                        object_reflect = True if object_xml.attrib["visiblespecular"] == "true" else False
                        object_shadow = True if object_xml.attrib["castshadow"] == "true" else False
                        object_atmospherics = True if object_xml.attrib["visiblevolume"] == "true" else False
                        
                        try:
                            ...
                            if object_name != base_object:
                                obj.data = bpy.data.objects[base_object].data
                        except:
                            ...
                        
                        if obj.type == "MESH":
                            obj.hide_render = not object_Render
                            # befor 3.00
                            if bpy.app.version < (3, 0, 0):
                                obj_vis = obj.cycles_visibility
                                obj_vis.camera = object_camera
                                # obj_vis.diffuse = False
                                obj_vis.glossy = object_reflect
                                obj_vis.transmission = object_reflect
                                obj_vis.scatter = object_atmospherics
                                obj_vis.shadow = object_shadow

                            # 3.00 later
                            else:
                                obj.visible_camera = object_camera
                                # obj.visible_diffuse = False
                                obj.visible_glossy = object_reflect
                                obj.visible_transmission = object_reflect
                                obj.visible_volume_scatter = object_atmospherics
                                obj.visible_shadow = object_shadow
                
                # Rename to original name
                for obj in bpy.data.objects:
                    ...
                    object_xml = xml_root.find("./geometries/*/[@name='%s']" % obj.name)
                    if (object_xml is None):
                        ...
                    else:
                        original_name = object_xml.attrib["original_name"]
                        obj.name = original_name

            except:
                print(traceback.print_exc())
            
            print("========= 模型加载完成 Ended =========")
        
        # 把所有物体移动到新的集合中
        maxfilename = xml_root.find("./productor/file").text
        if maxfilename == None:
            maxfilename = "Untitled"
        else:
            maxfilename = os.path.splitext(maxfilename)[0]
        # import_name, _ = os.path.splitext(xml_root.find("./productor/file").text)
        # Arch_Col = bpy.data.collections.new(import_name)
        Arch_Col = bpy.data.collections.new(maxfilename)
        bpy.context.collection.children.link(Arch_Col)

        objects_col = bpy.data.collections.new("Objects")
        Arch_Col.children.link(objects_col)

        for o in bpy.data.objects:
            objects_col.objects.link(o)
            bpy.context.collection.objects.unlink(o)

            if o.type == 'EMPTY' and o.children == ():
                bpy.data.objects.remove(o)

        # 设置场景
        if import_settings :
            renderWidth = int(xml_root.find("./settings/image_width").text)
            renderHeight = int(xml_root.find("./settings/image_height").text)
            bpy.context.scene.render.resolution_x = renderWidth
            bpy.context.scene.render.resolution_y = renderHeight

        # 设置相机
        if import_camera :
            camera_xml = xml_root.find("./cameras")

            camera_col = bpy.data.collections.new("Camera")
            Arch_Col.children.link(camera_col)

            for c in camera_xml.findall("./*") :
                try:
                    camera_name = c.attrib["original_name"]
                    camera_matrix = ConvertMatrix(c.attrib["transform"])
                    camera_fov = c.attrib["fov"]
                    #camera_clip = c.attrib["clip"]
                    camera_near = c.attrib["clippingnear"]
                    camera_far = c.attrib["clippingfar"]
                    camera_shiftX = float(c.attrib["shiftx"])
                    camera_shiftY = float(c.attrib["shifty"])
                    camera_sensorwidth = float(c.attrib["sensorwidth"])

                    camera_data = bpy.data.cameras.new(name='Cameras')
                    camera_object = bpy.data.objects.new(camera_name, camera_data)
                    camera_col.objects.link(camera_object)
                    # camera_object.data.angle = float(camera_fov)/180 * pi
                    camera_object.data.lens = float(camera_fov)
                    #if camera_clip == "true":
                    camera_object.data.clip_start = float(camera_near) * fac
                    camera_object.data.clip_end = float(camera_far) * fac
                    camera_object.data.sensor_fit = 'AUTO'
                    camera_object.data.sensor_width = camera_sensorwidth
                    camera_object.data.shift_x = camera_shiftX
                    camera_object.data.shift_y = camera_shiftY                   

                    camera_object.matrix_world = camera_matrix * fac
                    camera_object.scale = (1, 1, 1)
                except:
                    print(traceback.print_exc())

        # 设置灯光
        if import_light :
            print("========= 加载灯光设置 Loading... =========")
            light_col = bpy.data.collections.new("Lights")
            Arch_Col.children.link(light_col)
            light_xml = xml_root.find("./lights")

            base_object_list = [xx.attrib["base_object"] for xx in light_xml.findall("./*")]
            base_object_list = sorted(set(base_object_list))

            for bo in base_object_list:
                pass
                # 获取base object名称的参数（除了Matrix）
                base_object = light_xml.find("./*/[@name = '%s']" % bo)
                                
                L_Type = base_object.attrib["type"]
                IES = base_object.attrib["ies"]
                light_name = "tmp_light_name"
                color = ConvertColor(base_object.attrib["color"])

                if L_Type in ("AREA", "DISK", "RECTANGLE", "SQUARE"):
                    light_data = bpy.data.lights.new(name="Light", type = "AREA")
                else:
                    light_data = bpy.data.lights.new(name="Light", type = L_Type)

                light_obj = bpy.data.objects.new(light_name, light_data)
                light_obj.data.energy = float(base_object.attrib["strength"])
                light_obj.data.color = color[:-1]
                light_obj.data.cutoff_distance = 3

                if L_Type == "POINT":
                    light_obj.data.shadow_soft_size = 0.005 if float(base_object.attrib["length"]) * fac < 0.005 else float(base_object.attrib["length"]) * fac                        
                elif L_Type in ("RECTANGLE", "SQUARE") : #"AREA":
                    light_obj.data.shape = 'RECTANGLE'
                    light_obj.data.size = 0.005 if float(base_object.attrib["length"]) * fac < 0.005 else float(base_object.attrib["length"]) * fac
                    light_obj.data.size_y = 0.005 if float(base_object.attrib["width"]) * fac < 0.005 else float(base_object.attrib["width"]) * fac
                    light_obj.data.cycles.is_portal = True if base_object.attrib["portal"] == "true" else False
                elif L_Type == "DISK":
                    light_obj.data.shape = 'DISK'
                    light_obj.data.size = 0.005 if float(base_object.attrib["length"]) * fac < 0.005 else float(base_object.attrib["length"]) * fac
                    light_obj.data.size_y = 0.005 if float(base_object.attrib["width"]) * fac < 0.005 else float(base_object.attrib["width"]) * fac
                    light_obj.data.cycles.is_portal = True if base_object.attrib["portal"] == "true" else False
                elif L_Type == "SUN":
                    light_obj.data.angle = 0.005
                elif L_Type == "SPOT" :
                    pass

                ### Add IES File
                if IES != "":
                    light_obj.data.use_nodes = True

                    nodes = light_obj.data.node_tree.nodes
                    links = light_obj.data.node_tree.links

                    IES_Node = nodes.new("ShaderNodeTexIES")
                    IES_Node.mode = 'EXTERNAL'
                    IES_Node.filepath = IES
                    IES_Node.location = (-220, 300)
                    Emission_Node = nodes["Emission"]
                    links.new(IES_Node.outputs[0], Emission_Node.inputs["Strength"])

                # 查找所有同一Base Object属性的灯光，设置它们的Matrix
                base_object_instant = light_xml.findall("./*/[@base_object = '%s']" % bo)
                for light in base_object_instant:
                    ...
                    matrix = ConvertMatrix(light.attrib["transform"])

                    light_instance = light_obj.copy()
                    light_instance.name = light.attrib["original_name"]
                    light_instance.matrix_world = matrix * fac
                    light_instance.scale *= 1 / fac

                    light_instance.visible_camera = True if light.attrib["invisible"] == "false" else False
                    light_instance.visible_diffuse = True if light.attrib["diffuse"] == "true" else False
                    light_instance.visible_glossy = True if light.attrib["specular"] == "true" else False
                    light_instance.visible_transmission = True if light.attrib["reflection"] == "true" else False
                    
                    light_col.objects.link(light_instance)                  
                
                #Delete Origin objects
                bpy.data.objects.remove(light_obj)

            print("========= 灯光设置完成 Ended =========")

        # 设置代理模型
        '''
        if import_proxy :
            proxy_list = xml_root.find("./Proxy")

            proxy_col = bpy.data.collections.new("Proxy")
            Arch_Col.children.link(proxy_col)

            for p in proxy_list:
                try:
                    source_obj_name = p.attrib["Name"]
                    source_obj_file = p.attrib["filename"]
                    _, tempfilename = os.path.split(source_obj_file)
                    filename, extension = os.path.splitext(tempfilename)

                    _src_obj = bpy.data.objects[source_obj_name]
                    me = _src_obj.data
                    
                    empty_parent = bpy.data.objects.new((filename), None)
                    proxy_col.objects.link(empty_parent)

                    proxy_ins_item = p

                    for idx, item in enumerate(proxy_ins_item):
                        item_name = item.attrib["name"]
                        item_matrix = ConvertMatrix(item.attrib["matrix"])

                        _tmp_obj = bpy.data.objects.new(item_name, me)
                        _tmp_obj.display_type = "BOUNDS"
                        _tmp_obj.matrix_world = item_matrix * fac

                        proxy_col.objects.link(_tmp_obj)
                        _tmp_obj.parent = empty_parent

                    #Delete Origin objects
                    bpy.data.objects.remove(_src_obj)
                except:
                    print(traceback.print_exc())
            '''

        # 设置自动平滑
        if bpy.app.version < (4, 1, 0):
            for i in bpy.data.meshes:
                i.use_auto_smooth = True

        # 设置环境
        '''
        if import_environment :
            print("========= 加载环境设置 Loading... =========")

            env_color = ConvertColor(xml_root.find("./Environment/Color").text)
            env_intensity = float(xml_root.find("./Environment/Multiplier").text)
            env_texmap = xml_root.find("./Environment/Texmap")

            scene = bpy.context.scene
            world = scene.world
            world.use_nodes = True
            nodes = world.node_tree.nodes
            links = world.node_tree.links

            # clear default nodes
            nodes.clear()

            # create input node
            world_output_node = nodes.new("ShaderNodeOutputWorld")
            world_output_node.location = (0, 0)

            backgroud_node = nodes.new("ShaderNodeBackground")
            backgroud_node.location = (-200, 0)
            backgroud_node.inputs[0].default_value = env_color
            backgroud_node.inputs[1].default_value = env_intensity

            links.new(backgroud_node.outputs[0], world_output_node.inputs[0])

            # create texmap node
            if env_texmap :                
                if env_texmap.attrib["Texmap"] != "undefined":
                    texmap_type = env_texmap.attrib["Texmap"].split(":")[-1]
                    CreateNode(nodes, links, env_texmap[0], backgroud_node.inputs[0], texmap_type)
                
            print("========= 环境设置完成 Ended =========")
        
        '''

        # 清除缓存文件，防止下次导入时出现问题
        bpy.data.batch_remove(bpy.data.cache_files)

    if os.path.exists(MTLFile) and (os.path.exists(FBXFile) or os.path.exists(USDFile)):
        mtl_file = xml.parse(MTLFile)
        mtl_root = mtl_file.getroot()

        # 错误材质列表
        error_mat_list = []

        # 设置材质
        if import_material :
            print("========= 加载材质 Loading... =========")
            SceneMaterials = [ m for m in bpy.data.materials[:] if not m.is_grease_pencil]
            
            Create_ColorCorrection()

            for mat in SceneMaterials:
                material_name = mat.name
                nodes = mat.node_tree.nodes
                links = mat.node_tree.links
                
                mat.blend_method = "CLIP"

                try:
                    mat.shadow_method = "HASHED"
                except:
                    pass

                mat.metallic = 0

                m = mtl_root.find("./materials/*/[@name='%s']" % material_name)
                
                try:
                    nodes.clear()
                    outputShader = nodes.new("ShaderNodeOutputMaterial")
                    if hasattr(m, "tag"):
                        origin_name = m.attrib["original_name"]
                        
                        mat.name = origin_name
                        ### Double Materials
                        if m.tag == "vraydoublesidemtl" :
                            DoubleMixShader = nodes.new("ShaderNodeMixShader")
                            FrontBSDFShader = None
                            BackBSDFShader = None
                            TranslucentShader = None

                            links.new(DoubleMixShader.outputs["Shader"], outputShader.inputs["Surface"])
                            
                            frontMtl = m.attrib["frontmtl"].split(":")[0]
                            backMtl = m.attrib["backmtl"].split(":")[0]
                            mix_value = float(m.attrib["mix"])
                            
                            mask_map = m.attrib["mask"]                            
                            force1sidedsubmtls = True if m.attrib["force1sidedsubmtls"] == "true" else False
                            
                            # backmtl_on = m.attrib["backmtlon"]

                            DoubleMixShader.inputs["Fac"].default_value = mix_value

                            if frontMtl != "undefined":
                                FMat = mtl_root.find("./materials/singlemtl/[@name='%s']" % frontMtl)
                                FrontBSDFShader = CreateSingleMtl(nodes, links, FMat)
                                
                            if force1sidedsubmtls:
                                TranslucentShader = nodes.new("ShaderNodeBsdfTranslucent")
                                links.new(TranslucentShader.outputs["BSDF"], DoubleMixShader.inputs[2])
                                links.new(FrontBSDFShader.outputs["BSDF"], DoubleMixShader.inputs[1])
                                
                            else:
                                transparentShader = nodes.new("ShaderNodeBsdfTransparent")
                                links.new(transparentShader.outputs["BSDF"], DoubleMixShader.inputs[2])

                                DoubleMixBackShader = nodes.new("ShaderNodeMixShader")
                                links.new(DoubleMixBackShader.outputs["Shader"], DoubleMixShader.inputs[1])

                                backfacingNode = nodes.new("ShaderNodeNewGeometry")
                                for o in backfacingNode.outputs:
                                    if o.name != "Backfacing":
                                        o.hide = True
                                links.new(backfacingNode.outputs["Backfacing"], DoubleMixBackShader.inputs["Fac"])
                                links.new(FrontBSDFShader.outputs["BSDF"], DoubleMixBackShader.inputs[1])

                                if backMtl != "undefined":
                                    if backMtl == frontMtl:
                                        links.new(FrontBSDFShader.outputs["BSDF"], DoubleMixBackShader.inputs[2])
                                    else:
                                        BMat = mtl_root.find("./materials/singlemtl/[@Hash='%s']" % backMtl)
                                        BackBSDFShader = CreateSingleMtl(nodes, links, BMat)
                                        links.new(BackBSDFShader.outputs["BSDF"], DoubleMixBackShader.inputs[2])
                            
                            if mask_map != "undefined":
                                texmap_type = m.attrib["mask"].split(":")[-1]
                                CreateNode(nodes, links, m[0], DoubleMixShader.inputs["Fac"], texmap_type)

                                # Find base color of front mtl, and link to transnode                                
                                for lk in links:
                                    bc_from = None
                                    alpha_from = None
                                    if lk.to_socket.name == "Base Color":
                                        bc_from = lk.from_node
                                        links.new(bc_from.outputs[0], TranslucentShader.inputs["Color"])

                                    # Alpha ???
                                    if lk.to_socket.name == "Alpha":
                                        alpha_from = lk.from_node
                                        # 旧版MixRGB
                                        # if bpy.app.version < (3,5,0):
                                        # pass
                                        MixShader = nodes.new("ShaderNodeMixRGB")
                                        # MixShader = nodes.new("ShaderNodeMix")
                                        MixShader.blend_type = 'MULTIPLY'
                                        MixShader.inputs[0].default_value = 1

                                        links.new(alpha_from.outputs[0], MixShader.inputs["Color1"])
                                        if bc_from != None:
                                            links.new(bc_from.outputs[0], MixShader.inputs["Color2"])
                                        links.new(MixShader.outputs[0], TranslucentShader.inputs["Color"])
                        
                        elif m.tag == "doublemtl": 
                            DoubleMixShader = nodes.new("ShaderNodeMixShader")
                            FrontBSDFShader = None
                            BackBSDFShader = None
                            TranslucentShader = None

                            links.new(DoubleMixShader.outputs["Shader"], outputShader.inputs["Surface"])
                            
                            frontMtl = m.attrib["mtl1"].split(":")[0]
                            backMtl = m.attrib["mtl2"].split(":")[0]
                            mix_value = float(m.attrib["mix1"])
                            
                            mask_map = m.attrib["mask1"]                            

                            DoubleMixShader.inputs["Fac"].default_value = mix_value

                            if frontMtl != "undefined":
                                FMat = mtl_root.find("./materials/singlemtl/[@Hash='%s']" % frontMtl)
                                FrontBSDFShader = CreateSingleMtl(nodes, links, FMat)
                                links.new(FrontBSDFShader.outputs["BSDF"], DoubleMixShader.inputs[1])
                                
                            if backMtl != "undefined":
                                if backMtl == frontMtl:
                                    links.new(FrontBSDFShader.outputs["BSDF"], DoubleMixShader.inputs[2])
                                else:
                                    BMat = mtl_root.find("./materials/singlemtl/[@Hash='%s']" % backMtl)
                                    BackBSDFShader = CreateSingleMtl(nodes, links, BMat)
                                    links.new(BackBSDFShader.outputs["BSDF"], DoubleMixShader.inputs[2])
                            
                            if mask_map != "undefined":
                                texmap_type = m.attrib["mask1"].split(":")[-1]
                                CreateNode(nodes, links, m[0], DoubleMixShader.inputs["Fac"], texmap_type)

                        ### Single Material
                        elif m.tag == "singlemtl" :
                            node = CreateSingleMtl(nodes, links, m)
                            # TranslucecyNode
                            # 1.判断是不是有 Tanslucecy节点
                            # 2.判断是不是有 Alpha 节点，如果有的话，就把Alpha和Tanslucecy节点混合再连入Translucecy节点中
                            translucency_path = m.find("./translucent")
                            alpha_path = m.find("./opacity")
                            
                            if translucency_path.attrib["texmap"] != "undefined" :
                                # AddShader = nodes.new("ShaderNodeAddShader")
                                AddShader = nodes.new("ShaderNodeMixShader")
                                TranslucentShader = nodes.new("ShaderNodeBsdfTranslucent")
                                AddShader.inputs["Fac"].default_value = 0.4

                                links.new(AddShader.outputs["Shader"], outputShader.inputs["Surface"])
                                links.new(TranslucentShader.outputs["BSDF"], AddShader.inputs[2])
                                links.new(node.outputs["BSDF"], AddShader.inputs[1])

                                texmap_type = translucency_path.attrib["texmap"].split(":")[-1]

                                if alpha_path.attrib["texmap"] != "undefined" :
                                    # 旧版MixRGB
                                    MultiplyNode = nodes.new("ShaderNodeMixRGB")

                                    # 新版MixRGB
                                    # MultiplyNode = nodes.new("ShaderNodeMix")
                                    MultiplyNode.blend_type = 'MULTIPLY'

                                    # 新版Mix参数
                                    # MultiplyNode.data_type = 'RGBA'
                                    MultiplyNode.inputs[0].default_value = 1
                                    
                                    # 旧版Mix参数
                                    links.new(MultiplyNode.outputs["Color"], AddShader.inputs["Fac"])
                                    ## Add Mode
                                    links.new(MultiplyNode.outputs["Color"], TranslucentShader.inputs["Color"])
                                    # 新版Mix参数
                                    # links.new(MultiplyNode.outputs[2], TranslucentShader.inputs["Color"])

                                    for l in node.inputs["Alpha"].links:
                                        links.new(l.from_socket, MultiplyNode.inputs["Color2"])
                                        # links.new(l.from_socket, MultiplyNode.inputs[6])

                                    # 旧版Mix连接方式
                                    CreateNode(nodes, links, translucency_path[0], MultiplyNode.inputs["Color1"], texmap_type)
                                    # 新版Mix连接方式
                                    # CreateNode(nodes, links, translucency_path[0], MultiplyNode.inputs[7], texmap_type)
                                else:
                                    CreateNode(nodes, links, translucency_path[0], TranslucentShader.inputs["Color"], texmap_type)
                                    CreateNode(nodes, links, translucency_path[0], AddShader.inputs["Fac"], texmap_type)
                            else:
                                links.new(node.outputs["BSDF"], outputShader.inputs["Surface"])

                            # Displacement
                            Displacement_path = m.find("./displacement")
                            if Displacement_path.attrib["texmap"] != "undefined" :
                                DisplacementShader = nodes.new("ShaderNodeDisplacement")
                                DisplacementShader.mute = True
                                texmap_type = Displacement_path.attrib["texmap"].split(":")[-1]

                                CreateNode(nodes, links, Displacement_path[0], DisplacementShader.inputs["Normal"], texmap_type)
                                links.new(DisplacementShader.outputs["Displacement"], outputShader.inputs["Displacement"])
                            
                            # 禁用Displacement连线
                            for l in links:
                                if l.to_socket.name == "Displacement":
                                    l.is_muted = True

                        ### Light Material
                        elif m.tag == "lightmtl" :
                            emissionShader = nodes.new("ShaderNodeEmission")
                            
                            doubleSideShader = nodes.new("ShaderNodeMixShader")
                            mixTransShader = nodes.new("ShaderNodeMixShader")
                            TransparentShader = nodes.new("ShaderNodeBsdfTransparent")
                            GeometryShader = nodes.new("ShaderNodeNewGeometry")
                            MathShader = nodes.new("ShaderNodeMath")

                            links.new(mixTransShader.outputs["Shader"], outputShader.inputs["Surface"])
                            links.new(GeometryShader.outputs["Backfacing"], MathShader.inputs[0])
                            links.new(MathShader.outputs["Value"], doubleSideShader.inputs["Fac"])
                            links.new(emissionShader.outputs["Emission"], doubleSideShader.inputs[1])
                            links.new(doubleSideShader.outputs["Shader"], mixTransShader.inputs[1])
                            links.new(TransparentShader.outputs["BSDF"], doubleSideShader.inputs[2])
                            links.new(TransparentShader.outputs["BSDF"], mixTransShader.inputs[2])

                            emissionShader.location = (-750, 0)
                            mixTransShader.location = (-250, 0)
                            doubleSideShader.location = (-500, 0)
                            MathShader.location = (-750, 200)
                            GeometryShader.location = (-1000, 300)
                            TransparentShader.location = (-750, -200)

                            # 设置参数
                            emissionShader.inputs["Color"].default_value = ConvertColorMax(m.attrib["color"])
                            emissionShader.inputs["Strength"].default_value = float(m.attrib["multiplier"])

                            mixTransShader.inputs["Fac"].default_value = 0
                            MathShader.operation = 'MULTIPLY'
                            if m.attrib["twoSided"] == "false":
                                MathShader.inputs[1].default_value = 1
                            else :
                                MathShader.inputs[1].default_value = 0

                            if m.attrib["texmap"] != "undefined":                            
                                texmap_type = m.attrib["texmap"].split(":")[-1]
                                CreateNode(nodes, links, m[0][0], emissionShader.inputs["Color"], texmap_type)
                            
                            if m.attrib["opacity_texmap"] != "undefined" :
                                texmap_type = m.attrib["texmap"].split(":")[-1]
                                CreateNode(nodes, links, m[1][0], mixTransShader.inputs["Fac"], texmap_type)

                except Exception as ex:
                    print(traceback.print_exc())
                    error_mat_list.append(material_name)

            ## Set all images colorspace is Non-Color
            for i in bpy.data.images :
                i.colorspace_settings.name = "Non-Color" # "sRGB"  
            
            ## Set images ColorSapce
            img_nodes = []
            for mat in SceneMaterials:
                nodes = mat.node_tree.nodes
                links = mat.node_tree.links

                for i in nodes:
                    if i.type in ("BSDF_PRINCIPLED", "EMISSION"):
                        for n in ("Base Color", "Subsurface Color", "Emission", "Color"):
                            try:
                                nn = i.inputs[n]
                                if nn.is_linked:
                                    for l in nn.links:
                                        img_nodes.extend(CollectImagesNode(l.from_node))
                            except:
                                pass
            for i in img_nodes:
                if i.type == "TEX_IMAGE":
                    i.image.colorspace_settings.name = "sRGB"                    
        
            print("========= 材质加载完成 Ended =========")
            # 错误材质列表
            return error_mat_list
    else:
        return [0]

class OT_MaxFileBrower(Operator, ImportHelper):
    bl_idname = "redhalo.maxfilebrower"
    bl_label = "RedHaloM2B v." + __version__
    bl_space_type = 'PROPERTIES'
    bl_region_type = 'WINDOW'
    bl_options = {'UNDO'}
    
    filter_glob: StringProperty(
        default='*.max',
        options={'HIDDEN'}
    ) # type: ignore
    
    def execute(self, context):
        sc = context.scene
        settings = sc.redhalo_import_option

        pref = context.preferences
        addon_pref = pref.addons[__package__].preferences
        maxroot = addon_pref.MaxRootFolder

        script_file = os.path.realpath(__file__)
        directory = os.path.dirname(script_file)

        ini_file = os.path.join(directory, "RH_M2B.ini")
        version = bpy.app.version_string
        version = version.split(".")
        version_m = version[0] 
        version_b = version[1]

        # Write profile
        with open(ini_file, "w") as f:
            f.write("[RH_M2B]\n")
            f.write("main={}\n".format(version_m))
            f.write("branch={}\n".format(version_b))

            print(settings.import_format)

            if bpy.app.version < (3, 5, 0):
                f.write("format=FBX\n")
            else:
                f.write("format={}\n".format(settings.import_format))

            f.write("ungroup={}\n".format(settings.explode_group))
            f.write("ngon={}\n".format(settings.keep_ngon))

            f.close
        
        # Set New DateName option to False
        pref.view.use_translate_new_dataname = False
           
        maxcmd = maxroot + "3dsmaxcmd.exe"
        # maxcmd = maxroot + "3dsmaxbatch.exe"
        maxfile = self.filepath
        maxscript = os.path.join(directory, "export.ms")
        extra = '-script:"' + maxscript + '"' #"  -secure:0
        # extra = '-script:"' + maxscript + '"' #"  -secure:0
        doscmd = ('\""{}" "{}" {}\"').format(maxcmd, maxfile, extra)
        # doscmd = ('\""{}" "{}" -sceneFile {}\"').format(maxcmd, maxscript, maxfile)
        print(doscmd)
        if os.path.exists(maxcmd):
            ## Max Applicate Version
            MaxAppVersion = ""
            with open(maxcmd, "rb") as f:
                buff = f.read()
                inx = buff.find(b"\x01\x00\x46\x00\x69\x00\x6C\x00\x65\x00\x56\x00\x65\x00\x72\x00\x73\x00\x69\x00\x6F\x00\x6E\x00\x00\x00\x00\x00")
                f.seek(inx+28)
                temp = f.read(24).decode("utf-8")
                for i in range(0, len(temp), 2):
                    MaxAppVersion += temp[i]
                    
            MaxAppVersion = int(MaxAppVersion.split(".")[0])
            
            ## Max File Version
            MaxFileVerion = FindMax(maxfile)

            if MaxFileVerion > MaxAppVersion:
                self.report({'ERROR'}, "文件版本太高，请换一个更高版本的Max程序")
                return {'CANCELLED'}
            else:
                os.system(doscmd)
        else:
            self.report({'ERROR'}, "Max程序目录不正确，请重新指定,如果是手动输入的地址，注意路径结尾必须是'\\'")
            return {'CANCELLED'}        
        
        # Import Scene
        err = import_max_scene_operator(
            import_model = settings.import_model, 
            import_light = settings.import_light, 
            import_proxy = settings.import_proxy, 
            import_camera = settings.import_camera, 
            import_material = settings.import_material, 
            import_settings = settings.import_settings, 
            use_collection = settings.use_collection,
            toggle_cycles_render = settings.toggle_cycles_render
        )

        if len(err) > 0:
            if err[0] == 0:
                self.report({'ERROR'}, "文件无法转换，错误日志查看 %temp%\\RHM2B_log.log")
            else:
                self.report({'ERROR'}, "材质不能完整转换,详细列表看控制台")
    
            print("\t {} 个材质不能转换，名称如下".format(len(err)))
            for i,m in enumerate(err):
                print("\t{}/{}:{}".format(i+1, len(err),m))
              
        return {'FINISHED'}
    
    def draw(self, context):
        sc = context.scene
        settings = sc.redhalo_import_option

        # layout = self.layout
        
        # box = layout.box()
        # box.label(text = "依赖条件：")

class REDHALO_M2B_PREFS(AddonPreferences):
    bl_idname = __package__

    MaxRootFolder: StringProperty(
		name = "3dsMax Root",
		description = "select 3ds max install folder",
		subtype = 'DIR_PATH',
	)

    def draw(self, context):
        layout = self.layout
        box = layout.box()
        box.label(text = "尽量点击后面的按钮选择路径，如果要手动输入的地址，注意路径结尾必须是'\\'")
        box.prop(self, "MaxRootFolder")

        box = layout.box()
        box.label(text = "此插件依赖的max程序，只是做的后台自动导出和导入，因此max主程序还是要安装滴。")
        box.label(text = "关于渲染器：")
        box.label(text = "如果场景材质是VRay材质，那么必须安装VRay英文版")
        box.label(text = "如果场景材质是Corona材质，那么必须安装Corona英文版")
        box.label(text = "由于高版的VRay和Corona部分材质可以互相使用，因此建议2个渲染都安装")
        box.label(text = "其它：")
        box.label(text = "1.导出的文件位置系统的临时文件夹【%temp%】中")
        box.label(text = "2.对于安装3dsMax安装完成后C盘占用太大的，可以去控制面板把名字带有Autodesk Material相关的安装程序全部卸载.")
        
        box = layout.box()
        box.label(text = "支持的材质：")
        box.label(text = "VRay标准材质，VRay灯光材质，VRay双面材质，Corona标准材质，Corona物理材质，Corona灯光材质，标准材质")
        box.label(text = "支持的纹理")
        box.label(text = "Bitmap, Mix, AO, Brick(部分), Checker, Falloff(部分), ColorCorrection, RGB Multiply, Gradient, Normal, VRayBitmap, VrayColor, CoronaMix, CornaFrontBack等等")
        box.label(text = "支持的材质和纹理")

class OT_Import_Dialog(Operator, ImportHelper):
    bl_idname = "redhalo.dgs_import_settings"
    bl_label = "RedHalo Max to Blender : Import Settings"
    bl_description = "Select Import Method"
    bl_options = {"UNDO"}

    @classmethod
    def poll(cls, context):
        if bpy.app.version >= (3, 0, 0) and True:
            cls.poll_message_set('')
        return not False

    def execute(self, context):
        sc = context.scene
        settings = sc.redhalo_import_option
        
        if settings.onlyImport:
            
            # Import Scene            
            err = import_max_scene_operator(
                import_model = settings.import_model, 
                import_light = settings.import_light, 
                import_proxy = settings.import_proxy, 
                import_camera = settings.import_camera, 
                import_material = settings.import_material, 
                import_settings = settings.import_settings, 
                use_collection = settings.use_collection,
                toggle_cycles_render = settings.toggle_cycles_render
            )

            if len(err) > 0:
                if err[0] == 0:
                    self.report({'ERROR'}, "文件无法转换，错误日志位于 %temp%\\RH_M2B_TEMP 文件夹中")
                else:
                    self.report({'ERROR'}, "材质不能完整转换,详细列表看控制台")
        
                print("\t {} 个材质不能转换，名称如下".format(len(err)))
                for i,m in enumerate(err):
                    print("\t{}/{}:{}".format(i+1, len(err),m))
        else:
            bpy.ops.redhalo.maxfilebrower('INVOKE_DEFAULT')

        return {"FINISHED"}

    def draw(self, context):
        sc = context.scene
        settings = sc.redhalo_import_option

        layout = self.layout

        layout.label(text = "Import Options:")        
        box = layout.box()
        row = box.row(align=True)
        row.scale_y = 1.25
        row.prop(settings, "onlyImport", icon="IMPORT")
        
        row = box.row(align=True)
        row.scale_y = 1.25
        row.prop(settings, "import_format", expand = True)
        row.enabled = not settings.onlyImport

        layout.label(text = "Import Options:")
        box = layout.box()
        row = box.row(align=True)
        # row.scale_y = 1.25
        row.prop(settings, "import_model", icon="MESH_DATA")
        row.prop(settings, "import_light", icon="LIGHT_DATA")
        row.prop(settings, "import_proxy", icon="LINK_BLEND")
        row = box.row(align=True)
        row.scale_y = 1.25
        row.prop(settings, "import_camera", icon="CAMERA_DATA")
        row.prop(settings, "import_material", icon="MATERIAL")
        row.prop(settings, "import_settings", icon="PREFERENCES")
        
        layout.label(text = "Others:")
        box = layout.box()
        row = box.row(align=True)
        row.prop(settings, "toggle_cycles_render", icon="RESTRICT_RENDER_OFF")
        row.prop(settings, "explode_group", icon="OBJECT_HIDDEN")
        # row.prop(settings, "keep_ngon", icon="VIEW_ORTHO")

        box = layout.box()
        box.label(text = "依赖条件：", icon='ERROR')
        box.label(text = "1.3dsMax主程序（越高越好）")
        box.label(text = "2.对应材质相关的渲染器：")
        box.label(text = "  a.不支持VRay6.0以前的中文版  b.Corona英文版")

    def invoke(self, context, event):
        title = "Max2Blender IMPORTER(v" + __version__ + ")"
        return context.window_manager.invoke_props_dialog(self, width=300, title = title )

class RedHalo_Import_Option(PropertyGroup):
    import_model: BoolProperty(
        name = "Mesh",
        description = "Import Mesh",
        default = True
    ) # type: ignore

    import_light: BoolProperty(
        name = "Light",
        description = "Import Light",
        default = True
    ) # type: ignore

    import_proxy: BoolProperty(
        name = "Proxy",
        description = "Import Proxy",
        default = True
    ) # type: ignore

    import_camera: BoolProperty(
        name = "Camera",
        description = "Import Camera",
        default = True
    ) # type: ignore

    import_material: BoolProperty(
        name = "Material",
        description = "Import Material",
        default = True
    ) # type: ignore

    import_settings: BoolProperty(
        name = "Settings",
        description = "Import Settings",
        default = True
    ) # type: ignore

    use_collection: BoolProperty(
        name = "Use Collection",
        description = "Use Collection",
        default = True
    ) # type: ignore
    
    import_animate: BoolProperty(
        name = "Animate",
        description = "Import Animate",
        default = True
    ) # type: ignore
    
    onlyImport: BoolProperty(
        name = "仅导入从3ds Max端导出的模型",
        description = "仅导入从3ds Max端导出的模型或上次导入的模型",
        default = False
    ) # type: ignore

    import_format: EnumProperty(
        name = "Format",
        description = "Export Scene Type",
        #[(identifier, name, description, icon, number), ...]
        items=[("USD", "Fast", "导入速度最快，但是软件版本要求高", 'FORCE_FORCE', 1),
               ("FBX", "Slow", "兼容性最高，但是导入速度很慢", 'SORTTIME', 2),
        ],
        default="USD",
    ) # type: ignore

    toggle_cycles_render : BoolProperty(
        name = "Cycles Render",
        description = "Change to Cycles Render",
        default = True
    ) # type: ignore

    explode_group : BoolProperty(
        name = "Ungroup",
        description = "导出模型前，先把所有组分解",
        default = False
    ) # type: ignore

    keep_ngon : BoolProperty(
        name = "Ngon",
        description = "尽量保持Ngon结构",
        default = False
    ) # type: ignore

def menu_func_import(self, context):
    self.layout.operator("redhalo.dgs_import_settings", text="3ds Max File Importer(.max)", icon="EVENT_M")

classes = (
    OT_MaxFileBrower,
    REDHALO_M2B_PREFS,
    OT_Import_Dialog,
    RedHalo_Import_Option
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)
    bpy.types.TOPBAR_MT_file_import.append(menu_func_import)
    bpy.types.Scene.redhalo_import_option = bpy.props.PointerProperty(type = RedHalo_Import_Option)

def unregister():
    del bpy.types.Scene.redhalo_import_option
    bpy.types.TOPBAR_MT_file_import.remove(menu_func_import)    
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)